#include "multithreaded/awesomelockfree.hpp"
#include "util/shared.hpp"
#include <algorithm>
#include <iostream>
#include <iterator>
#include <sstream>
#include <unordered_set>

template <typename HashType> void combineHash(HashType& totalHash, HashType wordHash)
{
   totalHash = (totalHash << 2) + wordHash;
}

template <typename Data> bool updateAtomic(Data& a, Data expected, Data desired) {
   return reinterpret_cast<std::atomic<Data>&>(a).compare_exchange_weak(expected,desired);
}


void MTAwesomeLockfreeNgramer::query(const SequenceId queryVersion, const BorrowedString& document, QueryOut& out)
{
   auto& localState = threadStates[getThreadId()];
   auto& words = localState.words;
   auto& wordHashes = localState.wordHashes;

   words.reserve(document.size());
   wordHashes.reserve(words.size());

   wordHashes.emplace_back(0);
   words.emplace_back(document.data() - 1);
   tokenize::wordSplitOffsetsSSE(document, ' ', [this,&words,&wordHashes](const char* wordEnd) {
      words.emplace_back(wordEnd);
      const auto wordStart = words[words.size() - 2] + 1;
      auto word = BorrowedString(wordStart, wordEnd - wordStart);
      wordHashes.emplace_back(std::hash<BorrowedString>()(word));
   });

   for (size_t i = 1; i < words.size(); i++) {
      patterns.prefetchBucketPtr(wordHashes[i + 12]);
      patterns.prefetchBucketEntry(wordHashes[i + 6]);
      const auto wordStart = words[i - 1] + 1;
      auto offset = i;
      size_t suffixLength = 0;
      size_t minSuffixLength = 1;
      size_t numWords = 0;
      size_t skipLength = 1;
      HashType runningHash = 0;
      do {
         numWords++;
         const auto wordEnd = words[offset];
         const auto word = BorrowedString(wordStart, wordEnd - wordStart);
         combineHash<HashType>(runningHash, wordHashes[offset]);
         if (numWords == skipLength || numWords == minSuffixLength) {
            auto match = patterns.altFind(word, runningHash);
            if (match != nullptr) {
               auto& entry = *match;
               suffixLength = entry.longestSuffix;
               minSuffixLength = entry.shortestSuffix;
               skipLength <<= 1;
               if (entry.updateIfUnseen(queryVersion)) {
                  out.addNgram(word);
               }
            }
            else {
               break;
            }
         }
         offset++;
      } while (numWords < suffixLength && offset < words.size());
   }

   words.clear();
   wordHashes.clear();
   out.finishResult();
}

void MTAwesomeLockfreeNgramer::accountAdds(size_t){}
void MTAwesomeLockfreeNgramer::add(const SequenceId addVersion, const BorrowedString& ngram, __attribute__((unused))  const uint32_t patternId)
{
   auto& localState = threadStates[getThreadId()];
   Pool<ht_t::Entry>& entryPool = localState.entryPool;
   auto& words = localState.words;
   auto& wordHashes = localState.wordHashes;

   auto borrowedStoredNgram = localState.arena.intern(ngram);

   words.emplace_back(borrowedStoredNgram.data() - 1);
   wordHashes.emplace_back(0);
   {
      HashType runningHash = 0;
      tokenize::wordSplitOffsetsSSE(borrowedStoredNgram, ' ', [this,&words,&wordHashes,&runningHash](const char* prefix) {
         words.emplace_back(prefix);
         auto i = words.size() - 1;
         auto word = BorrowedString(words[i - 1] + 1, words[i] - words[i - 1] - 1);
         combineHash<HashType>(runningHash, std::hash<BorrowedString>()(word));
         wordHashes.emplace_back(runningHash);
         patterns.prefetchBucketPtr(runningHash);
      });
   }

   const uint16_t length = words.size() - 1;

   size_t i;
   uint64_t adds = 0;
   for (i = 1; i < words.size(); i++) {
      auto prefix = borrowedStoredNgram.substr(0, words[i] - borrowedStoredNgram.data());
      auto& info = *patterns.tryInsertLockfree(SmallString<>(prefix), wordHashes[i],entryPool, adds);
      if((info.longestSuffix < length) || (info.shortestSuffix > length && i != length)) {
         uint16_t storedLongestSuffix;
         uint16_t storedShortestSuffix;
         while ((storedLongestSuffix=info.longestSuffix) < length) {
            if(updateAtomic(info.longestSuffix,storedLongestSuffix,length)) {
               break;
            }
         }
         while ((storedShortestSuffix=info.shortestSuffix) > length && i != length) {
            if(updateAtomic(info.shortestSuffix,storedShortestSuffix,length)) {
               break;
            }
         }
      }
      if (i == length) {
         info.setVisibleVersion(addVersion);
      }
   }

   patterns.addToSize(adds);

   words.clear();
   wordHashes.clear();
}

void MTAwesomeLockfreeNgramer::remove(const SequenceId removeVersion, const BorrowedString& ngram)
{
   // Compute hash of word
   const char* start = ngram.data() - 1;
   HashType runningHash = 0;
   tokenize::wordSplitOffsets(ngram, ' ', [this, &start, &runningHash](const char* end) {
      auto word = BorrowedString(start + 1, end - start - 1);
      combineHash<HashType>(runningHash, std::hash<BorrowedString>()(word));
      start = end;
   });
   auto& localState = threadStates[getThreadId()];
   size_t adds=0;
   patterns.tryInsertLockfree(SmallString<>(ngram), runningHash,localState.entryPool,adds)->invalidate(removeVersion);
   patterns.addToSize(adds);
}

void MTAwesomeLockfreeNgramer::finalizeLoading() { patterns.grow(patterns.size() * 32); }

void MTAwesomeLockfreeNgramer::initThreadState(size_t numThreads) {
   threadStates.reserve(numThreads);
   for(unsigned i=0;i<numThreads;++i)
      threadStates.emplace_back();
   //threadStates.assign(numThreads, ThreadState());
}
