#include "util/hashtree.hpp"
#include "util/string.hpp"
#include <cstdint>
#include <gtest/gtest.h>

BorrowedString toBorrowed(const std::string& str)
{
   auto leakStr = new std::string(str);
   return strToBorrowed(*leakStr);
}

TEST(HashTree, BaseApi)
{
   auto split = [](const BorrowedString& value) {
      auto splits = value.split(' ');
      if (splits.second.size() > 0) {
         return std::make_pair(BorrowedString(splits.first.data(), splits.first.size()),
                               BorrowedString(splits.second.data() + 1, splits.second.size() - 1));
      }
      else {
         return splits;
      }
   };

   auto strA = toBorrowed("A B C");
   auto strB = toBorrowed("A B");
   auto strC = toBorrowed("A");
   auto strD = toBorrowed("");
   auto strE = toBorrowed("A B C D");
   auto strX = toBorrowed("B C D");

   auto hashtree = HashTree<BorrowedString, decltype(split)>(split);
   ASSERT_EQ(hashtree.query(strA), hashtree.end());
   ASSERT_EQ(hashtree.query(strD), hashtree.end());
   hashtree.add(strA);
   ASSERT_TRUE(hashtree.query(strA).second != hashtree.end().second);
   ASSERT_EQ(hashtree.query(strB).second, hashtree.end().second);
   ASSERT_EQ(hashtree.query(strC).second, hashtree.end().second);
   ASSERT_EQ(hashtree.query(strD).second, hashtree.end().second);
   ASSERT_EQ(hashtree.query(strE).second, hashtree.end().second);
   hashtree.add(strE);
   ASSERT_TRUE(hashtree.query(strA).second != hashtree.end().second);
   ASSERT_EQ(hashtree.query(strB).second, hashtree.end().second);
   ASSERT_EQ(hashtree.query(strC).second, hashtree.end().second);
   ASSERT_EQ(hashtree.query(strD).second, hashtree.end().second);
   ASSERT_TRUE(hashtree.query(strE).second != hashtree.end().second);
   hashtree.add(strC);
   ASSERT_TRUE(hashtree.query(strA).second != hashtree.end().second);
   ASSERT_EQ(hashtree.query(strB).second, hashtree.end().second);
   ASSERT_TRUE(hashtree.query(strC).second != hashtree.end().second);
   ASSERT_EQ(hashtree.query(strD).second, hashtree.end().second);
   ASSERT_TRUE(hashtree.query(strE).second != hashtree.end().second);
   hashtree.remove(strA);
   ASSERT_EQ(hashtree.query(strA).second, hashtree.end().second);
   ASSERT_EQ(hashtree.query(strB).second, hashtree.end().second);
   ASSERT_TRUE(hashtree.query(strC).second != hashtree.end().second);
   ASSERT_EQ(hashtree.query(strD).second, hashtree.end().second);
   ASSERT_TRUE(hashtree.query(strE).second != hashtree.end().second);

   ASSERT_TRUE(hashtree.query(strX, hashtree.query(strC).first) != hashtree.end());
}

TEST(HashTree, Matcher)
{
   auto split = [](const BorrowedString& value) {
      auto splits = value.split(' ');
      if (splits.second.size() > 0) {
         return std::make_pair(BorrowedString(splits.first.data(), splits.first.size()),
                               BorrowedString(splits.second.data() + 1, splits.second.size() - 1));
      }
      else {
         return splits;
      }
   };

   auto strA = toBorrowed("A B C");
   auto strB = toBorrowed("A B");
   auto strC = toBorrowed("A");
   auto strE = toBorrowed("A B C D A");

   auto str1 = toBorrowed("A");
   auto str2 = toBorrowed("B");
   auto str3 = toBorrowed("C");
   auto str4 = toBorrowed("D");

   auto hashtree = HashTree<BorrowedString, decltype(split)>(split);
   auto matcher = hashtree.matcher();
   matcher.match(str1);
   ASSERT_EQ(matcher.isMatch(), false);
   hashtree.add(strA);
   hashtree.add(strB);
   hashtree.add(strE);
   hashtree.add(strC);
   hashtree.remove(strA);
   matcher.match(str1);
   ASSERT_EQ(matcher.isMatch(), true);
   matcher.handleMatches([&strC](const BorrowedString& str) { ASSERT_EQ(str, strC); });
   matcher.match(str2);
   ASSERT_EQ(matcher.isMatch(), true);
   matcher.match(str3);
   ASSERT_EQ(matcher.isMatch(), false);
   matcher.match(str4);
   ASSERT_EQ(matcher.isMatch(), false);
   matcher.match(str1);
   ASSERT_EQ(matcher.isMatch(), true);
   {
      int matches = 0;
      matcher.handleMatches([&strE, &strC, &matches](const BorrowedString& str) {
         if (matches == 0) {
            ASSERT_EQ(str, strE);
         }
         else {
            ASSERT_EQ(str, strC);
         }
         matches++;
      });
      ASSERT_EQ(matches, 2);
   }
   matcher.match(str2);
   ASSERT_EQ(matcher.isMatch(), true);
   matcher.handleMatches([&strB](const BorrowedString& str) { ASSERT_EQ(str, strB); });
}