#include "multithreaded/interquery.hpp"
#include "util/shared.hpp"
#include <algorithm>
#include <iostream>
#include <iterator>
#include <sstream>
#include <tuple>
#include <vector>
#include <thread>
#include <unordered_set>
#include <util/prioritypool.hpp>


template <typename HashType> void combineHash(HashType& totalHash, HashType wordHash)
{
   totalHash = (totalHash << 2) + wordHash;
}

template <typename Data> bool updateAtomic(Data& a, Data expected, Data desired) {
   return reinterpret_cast<std::atomic<Data>&>(a).compare_exchange_weak(expected,desired);
}




void InterQAwesomeNgramer::calculateWordOffsetsAndHashes(std::vector<const char*>& words, std::vector<HashType>& wordHashes, const BorrowedString& document) 
{
   words.reserve(document.size());
   wordHashes.reserve(words.size());

   wordHashes.emplace_back(0);
   words.emplace_back(document.data() - 1);
   tokenize::wordSplitOffsetsSSE(document, ' ', [this,&words,&wordHashes](const char* wordEnd) {
      words.emplace_back(wordEnd);
      const auto wordStart = words[words.size() - 2] + 1;
      auto word = BorrowedString(wordStart, wordEnd - wordStart);
      wordHashes.emplace_back(std::hash<BorrowedString>()(word));
   });
}

void InterQAwesomeNgramer::intraQueryRange(const SequenceId queryVersion, const BorrowedString& document, QueryOut& queryOut, unsigned partition) {

   const bool singleThreaded = partition == PoolOperation::SINGLETHREADED_MARKER;

   auto& localState = threadQueryStates[getThreadId()];
   auto& seenPatterns = localState.seenPatterns;
   auto& foundPatterns = localState.foundPatterns;
   auto& words = localState.words;
   auto& wordHashes = localState.wordHashes;

   OutputVector& out = queryOut.outputVectors[partition];
   seenPatterns.resize(patternIdCounter+1,0); 


   calculateWordOffsetsAndHashes(words, wordHashes, document); 

   for (size_t i = 1; i < words.size(); i++) {
      const auto wordStart = words[i - 1] + 1;
      auto offset = i;
      if(!singleThreaded && wordHashes[offset]%QUERY_PARTITIONS != partition) continue;
      size_t numWords = 0;
      HashType runningHash = 0;
      uint16_t maxSuffixLength = 0;
      do {
         numWords++;

         const auto wordEnd = words[offset];
         const auto word = BorrowedString(wordStart, wordEnd - wordStart);
         combineHash<HashType>(runningHash, wordHashes[offset]);

         if (numWords == 1 || numWords <= maxSuffixLength) {
            auto match = patterns.altFind(word, runningHash);
            if (match != nullptr) {
               auto& entry = *match;
               if (entry.longestSuffix > maxSuffixLength)
                  maxSuffixLength = entry.longestSuffix;
               if (entry.isVisible(queryVersion) && !seenPatterns[entry.id]) {
                  seenPatterns[entry.id]=1;
                  foundPatterns.push_back(entry.id);
                  singleThreaded ? queryOut.addNgram(word) : out.emplace_back(i, word);
               }
            }
            else {
               if (numWords <= 5)
                  break;
            }

         }

         offset++;
      } while (numWords < maxSuffixLength && offset < words.size());
   }

   for(auto id : foundPatterns){
     seenPatterns[id]=0;
   }
   foundPatterns.clear();

   words.clear();
   wordHashes.clear();

   if (singleThreaded){
      queryOut.finishResult();
   }
}




void InterQAwesomeNgramer::query(const SequenceId queryVersion, const BorrowedString& document, QueryOut& out)
{

   auto& localState = threadQueryStates[getThreadId()];
   auto& seenPatterns = localState.seenPatterns;
   auto& foundPatterns = localState.foundPatterns;
   auto& words = localState.words;
   auto& wordHashes = localState.wordHashes;

   seenPatterns.resize(patternIdCounter+1,0);


   calculateWordOffsetsAndHashes(words, wordHashes, document);

   for (size_t i = 1; i < words.size(); i++) {
      patterns.prefetchBucketPtr(wordHashes[i + 12]);
      patterns.prefetchBucketEntry(wordHashes[i + 6]);
      const auto wordStart = words[i - 1] + 1;
      auto offset = i;
      size_t suffixLength = 0;
      size_t minSuffixLength = 1;
      size_t numWords = 0;
      size_t skipLength = 1;
      HashType runningHash = 0;
      do {
         numWords++;
         const auto wordEnd = words[offset];
         const auto word = BorrowedString(wordStart, wordEnd - wordStart);
         combineHash<HashType>(runningHash, wordHashes[offset]);
         // std::cerr<<minSuffixLength<<"::"<<word.toStr()<<"::"<<numWords<<std::endl;
         //if (numWords == skipLength || numWords == minSuffixLength) {
            auto match = patterns.altFind(word, runningHash);
            if (match != nullptr) {
               auto& entry = *match;
               suffixLength = entry.longestSuffix;
               minSuffixLength = entry.shortestSuffix;
               skipLength <<= 1;
               if (entry.isVisible(queryVersion) && !seenPatterns[entry.id]) {
                  seenPatterns[entry.id]=1;
                  foundPatterns.push_back(entry.id);
                  out.addNgram(word);
               }
            }
            else {
               break;
            }
         //}
         offset++;
      } while (numWords < suffixLength && offset < words.size());
   }
   out.finishResult();

   for(auto id : foundPatterns){
     seenPatterns[id]=0;
   }
   foundPatterns.clear();

   words.clear();
   wordHashes.clear();
}

void InterQAwesomeNgramer::accountAdds(size_t adds){patterns.addToSize(adds);}

void InterQAwesomeNgramer::add(const SequenceId addVersion, const BorrowedString& ngram,size_t& adds, const uint32_t patternId)
{
   auto& localState = threadStates[getThreadId()];
   Pool<ht_t::Entry>& entryPool = localState.entryPool;

   const size_t MAX_SUBPATTERN_LENGTH=5;
   HashType wordHashes[MAX_SUBPATTERN_LENGTH+1];
   const char* words[MAX_SUBPATTERN_LENGTH+1];

   auto borrowedStoredNgram = SmallString<char>::needsAllocation(ngram) ?  localState.arena.intern(ngram) :  ngram;
#if 1
   words[0]=borrowedStoredNgram.data() - 1;
   wordHashes[0]=0;
   HashType fullHash=0;
   size_t count=0;
   {
      HashType runningHash = 0;
      const char* lastWord=words[0];
      tokenize::wordSplitOffsetsSSE(borrowedStoredNgram, ' ', [this,&words,&wordHashes,&runningHash,&count,&lastWord,MAX_SUBPATTERN_LENGTH](const char* prefix) {
         auto word = BorrowedString(lastWord + 1, prefix - lastWord - 1);
         combineHash<HashType>(runningHash, std::hash<BorrowedString>()(word));
         lastWord = prefix;
         count++;
         if(count<(MAX_SUBPATTERN_LENGTH+1)) {
            wordHashes[count]=runningHash;
            words[count]=prefix;
         }
      });
      fullHash=runningHash;
   }

   const uint16_t length = count;

   // Index first word
   auto prefix = borrowedStoredNgram.substr(0, words[1] - borrowedStoredNgram.data());
   auto& info = *patterns.tryInsertLockfree(SmallString<>(prefix), wordHashes[1],entryPool, adds);

   uint16_t storedLongestSuffix;
   while ((storedLongestSuffix = info.longestSuffix) < length) {
      ((std::atomic<uint16_t>*)&(info.longestSuffix))->compare_exchange_strong(storedLongestSuffix, length);
   }

   if (length == 1) {
      info.setVisibleVersion(addVersion);
      uint32_t expectedId=AwesomePatternInfo::INIT_ID;
      auto id = (std::atomic<uint32_t>*)&info.id;
      // in case of failure, someone else wrote the id first
      // thus, the pattern already has an id. no need to retry
      id->compare_exchange_strong(expectedId, patternId);
   } else {

       for (size_t i = 2; i <= MAX_SUBPATTERN_LENGTH && i < length; i++) {
           auto prefix = borrowedStoredNgram.substr(0, words[i] - borrowedStoredNgram.data());
           auto& info = *patterns.tryInsertLockfree(SmallString<>(prefix), wordHashes[i],entryPool, adds);
       }

      auto prefix = borrowedStoredNgram;
      auto& info = *patterns.tryInsertLockfree(SmallString<>(prefix), fullHash,entryPool, adds);
      info.setVisibleVersion(addVersion);
      uint32_t expectedId=AwesomePatternInfo::INIT_ID;
      auto id = (std::atomic<uint32_t>*)&info.id;
      // in case of failure, someone else wrote the id first
      // thus, the pattern already has an id. no need to retry
      id->compare_exchange_strong(expectedId, patternId);
   }

#else
   words.emplace_back(borrowedStoredNgram.data()- 1);
   {
      HashType runningHash = 0;
      tokenize::wordSplitOffsetsSSE(borrowedStoredNgram, ' ', [this,&words,&wordHashes,&runningHash](const char* prefix) {
         words.emplace_back(prefix);
         auto i = words.size() - 1;
         auto word = BorrowedString(words[i - 1] + 1, words[i] - words[i - 1] - 1);
         combineHash<HashType>(runningHash, std::hash<BorrowedString>()(word));
         wordHashes.emplace_back(runningHash);
         //patterns.prefetchBucketPtr(runningHash);
      });
   }

   const uint16_t length = words.size() - 1; 
   
   int32_t n=wordHashes.size()-1;
   auto prefix = borrowedStoredNgram.substr(0, words.back() - borrowedStoredNgram.data());
   auto info = patterns.tryInsertLockfree(SmallString<>(prefix), wordHashes.back(),entryPool,adds);
   info->setVisibleVersion(addVersion);
	uint32_t expectedId=AwesomePatternInfo::INIT_ID;
	auto id = (std::atomic<uint32_t>*)&info->id;
	// in case of failure, someone else wrote the id first
	// thus, the pattern already has an id. no need to retry
	id->compare_exchange_strong(expectedId, patternId);

   for(--n; n >=0;--n){
     auto prefix = borrowedStoredNgram.substr(0, words[n+1] - borrowedStoredNgram.data());
     info = patterns.tryInsertLockfree(SmallString<>(prefix),wordHashes[n],entryPool,adds);
     if(!info->longestSuffix){
       info->longestSuffix=length;
     } else {
       break;
     }
   } 

#endif
}

void InterQAwesomeNgramer::remove(const SequenceId removeVersion, const BorrowedString& ngram, size_t& adds)
{
   // Compute hash of word
   const char* start = ngram.data() - 1;
   HashType runningHash = 0;
   tokenize::wordSplitOffsets(ngram, ' ', [this, &start, &runningHash](const char* end) {
      auto word = BorrowedString(start + 1, end - start - 1);
      combineHash<HashType>(runningHash, std::hash<BorrowedString>()(word));
      start = end;
   });
   auto& localState = threadStates[getThreadId()];

   patterns.tryInsertLockfree(SmallString<>(ngram), runningHash,localState.entryPool, adds)->invalidate(removeVersion);
}


void InterQAwesomeNgramer::finalizeLoading() {
   patterns.grow(patterns.size() * 32);
}

void InterQAwesomeNgramer::initThreadState(size_t nT) {
   numThreads = nT;
   threadQueryStates.resize(numThreads);
   threadStates.resize(numThreads);
}
