/*
Copyright 2013 Henrik Mühe and Florian Funke

This file is part of CampersCoreBurner.

CampersCoreBurner is free software: you can redistribute it and/or modify
it under the terms of the GNU Affero General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

CampersCoreBurner is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU Affero General Public License for more details.

You should have received a copy of the GNU Affero General Public License
along with CampersCoreBurner. If not, see <http://www.gnu.org/licenses/>.
*/

/*
COMMENT SUMMARY FOR THIS FILE:

This implements a functor class which wraps the intrinsic crc32 instructions
inside modern cpus and exposes them to compute hashes. Since for this application
we have a known length limit on strings, we do not need a loop to compute a
string has but instead only a switch between different sizes classes.
*/

#pragma once

#include <cstdint>
#include <string>
#include <tuple>
#include <x86intrin.h>

namespace campers {

inline uint32_t larsonHash(uint32_t x)
{
   const char* xArray = reinterpret_cast<const char*>(&x);
   unsigned h = 0;
   for (unsigned i = 0; i < 4; ++i)
      h = h * 101 + (unsigned)*xArray++;
   return h;
}

inline uint32_t larsonHash(const char* x, uint32_t len)
{
   unsigned h = 0;
   for (unsigned i = 0; i < len; ++i)
      h = h * 101 + (unsigned)*x++;
   return h;
}

constexpr const uint32_t seed = 0x9f462312;

/// Wraps crc32 cpu instructions so that they can be used with unordered_map or our custom hash
class CampersHash {
   public:
   using HashType = uint32_t;
   /// Integers
   inline HashType operator()(uint64_t x) const { return std::hash<decltype(x)>()(x); }
   inline HashType operator()(int64_t x) const { return std::hash<decltype(x)>()(x); }
   inline HashType operator()(uint32_t x) const { return std::hash<decltype(x)>()(x); }
   inline HashType operator()(int32_t x) const { return std::hash<decltype(x)>()(x); }

   /// Pointers
   inline HashType operator()(void* ptr) const { return std::hash<void*>()(ptr); }

   template <typename X> HashType operator()(const X& x) const { return std::hash<X>()(x); }
   template <typename X, typename Y> HashType operator()(const std::pair<X, Y>& x) const
   {
      return CampersHash()(x.first) + CampersHash()(x.second);
   }
};

/// Wraps crc32 cpu instructions so that they can be used with unordered_map or our custom hash
class CampersHash64 {
   public:
   using HashType = size_t;
   /// Integers
   inline HashType operator()(uint64_t x) const { return std::hash<decltype(x)>()(x); }
   inline HashType operator()(int64_t x) const { return std::hash<decltype(x)>()(x); }
   inline HashType operator()(uint32_t x) const { return std::hash<decltype(x)>()(x); }
   inline HashType operator()(int32_t x) const { return std::hash<decltype(x)>()(x); }

   /// Pointers
   inline HashType operator()(void* ptr) const { return std::hash<void*>()(ptr); }

   template <typename X> HashType operator()(const X& x) const { return std::hash<X>()(x); }
   template <typename X, typename Y> HashType operator()(const std::pair<X, Y>& x) const
   {
      return CampersHash()(x.first) + CampersHash()(x.second);
   }
};
}
