#include <iostream>
#include <fstream>
#include <unordered_map>
#include <vector>
#include <map>
#include <set>
#include <algorithm>
#include <chrono>
#include <cstring>
#include "NeumannMatcher.hpp"
#include "StringHelper.hpp"
#include "Hash.hpp"

using namespace std;

static uint64_t hashString(const char* str,unsigned len)
{
#if 1
  return crc32_hash_rowdy(str, len);
#elif 0
   if (len<8) {
      uint64_t v=0;
      if (len>=4) { v=*reinterpret_cast<const uint32_t*>(str); str+=4; len-=4; }
      if (len>=2) { v=(v<<16)|(*reinterpret_cast<const uint16_t*>(str)); str+=2; len-=2; }
      if (len>=1) { v=(v<<8)|(*reinterpret_cast<const uint8_t*>(str)); }
      return v*11400714819323198485ull;
   } else {
      return (*reinterpret_cast<const uint64_t*>(str))*11400714819323198485ull;
   }
#elif 0
   constexpr uint64_t prime=0x100000001b3;
   uint64_t hash=0xcbf29ce484222325;
   if (len<8) {
      uint64_t v=0;
      if (len>=4) { v=*reinterpret_cast<const uint32_t*>(str); str+=4; len-=4; }
      if (len>=2) { v=(v<<16)|(*reinterpret_cast<const uint16_t*>(str)); str+=2; len-=2; }
      if (len>=1) { v=(v<<8)|(*reinterpret_cast<const uint8_t*>(str)); }
      hash^=v;
      return hash*prime;
   } else {
      while (len>=8) {
         hash^=*reinterpret_cast<const uint64_t*>(str);
         hash*=prime;
         str+=8; len-=8;
      }
      if (len) {
         unsigned ofs=8-len;
         hash^=(*reinterpret_cast<const uint64_t*>(str-ofs))>>ofs;
         hash*=prime;
      }
      return hash;
   }
#else
   return MurmurHash64A(str,len,0);
#endif
}

static inline uint64_t hashString(const string_view& s) { return hashString(s.data(),s.length()); }


vector<string_view> split(string_view& s){
    vector<string_view> result;
    auto start = s.data();
    for(auto dataEnd = start + s.size(); start < dataEnd;){
      auto end=firstWhitespace(start, dataEnd);
      result.emplace_back(start, end-start);
      start = end + 1;
    }
    return result;
}

void NeumannMatcher::addPattern(string_view& line)
{
   auto parts=split(line);
   if (parts.size()==1) {
     hashTable.tryInsert(parts[0])->match=true;
   } else {
      uint64_t rolling=0;
      for (unsigned index=0,limit=parts.size()-1;index!=limit;++index)
         rolling=(rolling>>8)|(hashString(parts[index])&(~((~0ull)>>8)));
      auto& e=*hashTable.tryInsert(parts.back());

      unsigned len=parts.size()-1;
      if (len>8) len=8;
      auto iter=lower_bound(e.patterns.begin(),e.patterns.end(),rolling,[len,&line](const NeumannMatcher::Pattern& a,uint64_t b) { if (a.hash!=b) return a.hash<b; if (a.len!=len) return a.len<len; return a.pattern<line; });
      if (iter!=e.patterns.end()) {
         // Already known?
         if ((iter->hash==rolling)&&(iter->len==len)&&(iter->pattern==line))
            return;
      }
      e.patterns.insert(iter,Pattern{rolling,line,len});
   }
}

void NeumannMatcher::deletePattern(string_view& line)
{
   auto parts=split(line);
   if (parts.size()==1) {
      auto p = hashTable.find(parts[0]);
      if(p) p->match=false;
   } else {
      uint64_t rolling=0;
      for (unsigned index=0,limit=parts.size()-1;index!=limit;++index)
         rolling=(rolling>>8)|(hashString(parts[index])&(~((~0ull)>>8)));
      auto p =hashTable.find(parts.back());
      if(!p) return;
      auto& e=*p;

      unsigned len=parts.size()-1;
      if (len>8) len=8;
      auto iter=lower_bound(e.patterns.begin(),e.patterns.end(),rolling,[len,&line](const NeumannMatcher::Pattern& a,uint64_t b) { if (a.hash!=b) return a.hash<b; if (a.len!=len) return a.len<len; return a.pattern<line; });
      if (iter!=e.patterns.end()) {
         if ((iter->hash==rolling)&&(iter->len==len)&&(iter->pattern==line))
            e.patterns.erase(iter,iter+1);
      }
   }
}


void NeumannMatcher::checkMatch(vector<tuple<unsigned,unsigned,const string_view*>>& hits, NeumannMatcher::Pattern& pattern,const char* begin,const char* limit,unsigned queryId)
   // Check if we have a match
{
   if (pattern.lastHit==queryId)
      return;

   checks++;
   unsigned patLen=pattern.pattern.length();
   if (patLen<=(limit-begin)) {
      if ((memcmp(limit-patLen,pattern.pattern.data(),patLen)==0)&&((patLen==(limit-begin))||((limit-patLen)[-1]==' '))) {
         hits.emplace_back(static_cast<unsigned>(limit-patLen-begin),patLen,&pattern.pattern);
         pattern.lastHit=queryId;
         hitCount++;
      }
   }
}

static inline uint64_t getFirst(uint64_t v,unsigned len) { return (len<8)?(v&(~((~0ull)>>(8*len)))):v; }

//====================================================================
void NeumannMatcher::query(string_view line)noexcept
{

  vector<tuple<unsigned,unsigned,const string_view*>> hits;


  const char* start = line.data();
  vector<const char*> words;
  vector<uint64_t> hashWords;
  words.reserve(INIT_SIZE_QUERY_VECTORS);
  hashWords.reserve(INIT_SIZE_QUERY_VECTORS);

  const char* end = start+line.size();

  //get word offsets and prefetch hash values
  words.push_back(start-1);
  for(auto curPos=start;curPos<end;){
    auto pos=firstWhitespace(curPos,end);
    auto length = pos-curPos;
    hashWords.push_back(hash(curPos,length));
    words.push_back(pos);
    curPos=pos+1;
  }
  const uint32_t lastOffset =words.size()-1;


  uint64_t rolling;
  rolling=0; ++queryId;
  //match words
  for(uint32_t wordPos=0; wordPos < lastOffset;++wordPos){
    hashTable.prefetchDir(hashWords[wordPos+20]);
    hashTable.prefetchFirst(hashWords[wordPos+10]);
    const char* wordStart=words[wordPos]+1;
    auto wordEnd = words[wordPos+1];
    string_view word(wordStart, wordEnd-wordStart);
    ++wordCount;
    auto iter2=hashTable.findEntry(word, hashWords[wordPos]);
    if (iter2) {
      auto& e=*iter2;
      ++candidates;
      if (e.value.match&&(e.value.lastHit!=queryId)) {
        hits.emplace_back(static_cast<unsigned>(wordStart-line.data()),1u,&e.word);
        e.value.lastHit=queryId;
      }
      // no binary search for small vectors
      if(e.value.patterns.size()<8){
        for(auto& p: e.value.patterns){
          if(p.hash==getFirst(rolling,p.len)){
            checkMatch(hits,p, line.data(), wordEnd, queryId);
          }
        }
      }else{
        auto lower=lower_bound(e.value.patterns.begin(),e.value.patterns.end(),rolling&(~((~0ull)>>8)),[](const NeumannMatcher::Pattern& a,uint64_t b) { return a.hash<b; });
        auto upper=upper_bound(lower,e.value.patterns.end(),rolling,[](uint64_t a,const NeumannMatcher::Pattern& b) { return a<b.hash; });
        unsigned len=1;
        while (lower<upper) {
          if (lower->hash==getFirst(rolling,lower->len)) {
            checkMatch(hits,*lower,line.data(),wordEnd,queryId);
            ++lower;
          } else {
            ++len;
            unsigned diff=__builtin_clzll(lower->hash^rolling)>>3;
            if (diff>len) len=diff;
            lower=lower_bound(lower+1,upper,rolling&(~((~0ull)>>(8*len))),[](const NeumannMatcher::Pattern& a,uint64_t b) { return a.hash<b; });
          }
        }
      }
    }
    rolling=(rolling>>8)|(hashString(word)&(~((~0ull)>>8)));
    wordStart = wordEnd + 1;
  }
  const char notFound[]="-1";
  const char lineEnd[]="\n";
  const char pipe[]="|";
  if (hits.empty()) {
        fwrite(notFound, 1, 2, stdout);
        fwrite(lineEnd, 1, 1, stdout);
        return;
  } else {
    sort(hits.begin(),hits.end());
    bool first=true;
    for (auto& h:hits) {
      if (first) first=false; else fwrite(pipe,1,1,stdout);
      auto p = get<2>(h);
      fwrite(p->data(),1,p->size(),stdout);
    }
    fwrite(lineEnd, 1, 1, stdout);
  }
}

//====================================================================
void NeumannMatcher::add(const string_view ngram)noexcept
{
  //XXX memory leak, this is not cleaned up
  char* loc = new char[ngram.size()];
  ngram.copy(loc, ngram.size()-1);
  string_view pattern(loc, ngram.size()-1);
  addPattern(pattern);
}

//====================================================================
void NeumannMatcher::remove(const string_view ngram)noexcept
{
  string_view pattern(ngram);
  pattern.remove_suffix(1);
  deletePattern(pattern);
}

NeumannMatcher::NeumannMatcher():hashTable(100000){}

NeumannMatcher::~NeumannMatcher(){
  cerr << "Words: " << wordCount << " candidates: " << candidates << " full checks: " << checks << " hits: " << hitCount << endl;
}
