/*
Copyright 2013 Henrik Mühe and Florian Funke

This file is imported from CampersCoreBurner.

CampersCoreBurner is free software: you can redistribute it and/or modify
it under the terms of the GNU Affero General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

CampersCoreBurner is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU Affero General Public License for more details.

You should have received a copy of the GNU Asffero General Public License
along with CampersCoreBurner. If not, see <http://www.gnu.org/licenses/>.
*/

/*
COMMENT SUMMARY FOR THIS FILE:

This file contains an implementation of an optimized chaining hashset and hashmap.
Depending on the use case, these hashes perform up to 4x better than std::unordered_map.
*/

#pragma once

#include "pool.hpp"
#include <cstdint>
#include <iostream>
#include <numeric>
#include <vector>

namespace campers {

/// A fast, chaining HashMap
template<class T,class Value,class HashFunc>
class HashMap {
    /// An entry inside the hash map
public:
    struct Entry {
        /// The key
        T word;
        /// The value for this key
        Value value;
        /// The hash value
        uint64_t hashValue;
        /// Pointer to chain of entries for collisions
        Entry* next;

        /// Constructor
        Entry () : hashValue(0),next(nullptr) {}
        /// Constructor
        Entry (const T& word,uint64_t hashValue) : word(word),hashValue(hashValue),next(nullptr) {}
        /// Constructor
        Entry (const T& word,uint64_t hashValue,Entry* next) : word(word),hashValue(hashValue),next(next) {}
        // Lazy Constructor
        template<class T2> Entry (const T2& word,uint64_t hashValue) : word(word),hashValue(hashValue),next(nullptr) {}
    };

    /// The hashmap of pointers to actual entries
    std::vector<Entry*> entries;

public:
    /// An instance of the hash funtion functor
    const HashFunc hashFunc;
    uint64_t collisionCounter=0;
    uint64_t lookupCounter=0;
private:
    /// The pool allocator.
    Pool<Entry> entryAllocator;
    /// The exponent, see HashSet and summary for details
    uint64_t exp;
    /// The mask, see HashSet and summary for details
    uint64_t mask;
    /// The number of elements inside the hash,
    uint64_t size_;
    /// The size this hash map was allocated with
    uint64_t allocSize;
    /// Maximum fill of hashtable before growing
    static constexpr float LoadFactor = 2;

public:
    /// Constructor
    HashMap(uint64_t size): hashFunc() { hintSize(size); }
    /// Inserts an element into the hashtable. Returns the value for that key, regardless if an insertion happened or not. Hash value supplied externally
    Value* tryInsert(const T& word,uint64_t hashValue);
    /// Inserts an element into the hashtable. Returns the value for that key, regardless if an insertion happened or not.
    Value* tryInsert(const T& word) { return tryInsert(word,hashFunc(word)); }
    /// Inserts an element into the hashtable. Returns the value for that key, regardless if an insertion happened or not.
    template <class T2> Value* tryAltInsert(const T2& word,uint64_t hashValue);
    /// Inserts an element into the hashtable. Returns the value for that key, regardless if an insertion happened or not.
    //Only call this method if you are sure, that the value is not already present
    Value* forceInsert(const T& word,uint64_t hashValue);
    /// Finds an element inside the hashmap, returns the value if found or nullptr otherwise. Hash value supplied externally
    Value* find(const T& word,uint64_t hashValue);
    /// Finds an element inside the hashmap, returns the value if found or nullptr otherwise
    Value* find(const T& word) { return find(word,hashFunc(word)); }
    /// Find which exposes key and value
    Entry* findEntry(const T& word) { return altFindEntry(word,hashFunc(word)); }
    /// Find which exposes key and value
    Entry* findEntry(const T& word,uint64_t hashValue) { return altFindEntry(word,hashValue); }
    /// Alternative find: uses a type that can differ from T, e.g. T=string, T2=string_ref
    template <class T2> Value* altFind(const T2& word, uint64_t hashValue);
    /// Alternative find: uses a type that can differ from T, e.g. T=string, T2=string_ref. Hash value supplied externally
    template <class T2> Value* altFind(const T2& word) { return altFind(word, hashFunc(word));}
    /// Alternative find which exposes key and value
    template <class T2> Entry* altFindEntry(const T2& word, uint64_t hashValue) ;
    /// Issue prefetch instruction to directory position of given hashValue
    void prefetchDir(uint64_t hashValue);
    /// Issue prefetch to first entry for given hash
    void prefetchFirst(uint64_t hashValue);
    /// Pre allocates the hash to be able to contain size elements efficiently. Also, see summary and HashSet
    void hintSize(uint64_t size);
    /// Returns the current number of elements inside the hash
    uint64_t size() const { return size_; }
    /// Grows the hash to twice its capacity
    void grow() { grow(2*capacity()); };
    /// Grows the hash to be able to efficiently contain 2* capacity elements
    void grow(uint64_t newSize);
    /// Clears all entries in the HashMap
    void clear();
    /// Returns the current capacity of the hash
    size_t capacity() const { return entries.size(); };

    /// Returns 1 if the value is contained
    size_t count(const T& word) { return find(word)!=nullptr?1:0; }
};
/// Grows the hash to be able to efficiently contain 2* capacity elements
template <class T, class Value, class HashFunc>
void HashMap<T, Value, HashFunc>::grow(uint64_t newSize){
   allocSize = newSize;
   // Grow
   for (exp = 1; (uint64_t)(1 << exp) < newSize; ++exp) {
   }
   // std::cerr << "Allocating " << allocSize << "::" << exp << "::" << (1 << exp) << std::endl;
   std::vector<Entry*> newEntries(1 << exp);
   mask = (1ull << exp) - 1ull;

   // Copy entries over
   for (Entry* entryChain : entries) {
      for (Entry *e = entryChain, *next; e != nullptr; e = next) {
         Entry*& entry = newEntries[e->hashValue & mask];
         if (entry == nullptr) { // immediate hit: use slot for e
            entry = e;
            next = e->next;
            e->next = nullptr;
         }
         else { // slot already in use: prepend e to chain
            next = e->next;
            e->next = entry;
            entry = e;
         }
      }
   }

   // Switch to new entry vector
   entries = std::move(newEntries);
}

/// Inserts an element into the hashtable. Returns the value for that key, regardless if an insertion happened or not.
template<class T,class Value,class HashFunc>
Value* HashMap<T,Value,HashFunc>::tryInsert(const T& word,uint64_t hashValue) {
    if(capacity() <= (size_t)(size() * LoadFactor)){
        grow();
    }
    Entry*& entry=entries[hashValue&mask];
    // Unroll empty slot branch
    if (entry==nullptr) {
        ++size_;
        entry=entryAllocator.allocate();
        new (entry) Entry(word,hashValue);
        return &entry->value;
    } else {
        // Unroll immediate hit branch
        if (entry->hashValue==hashValue&&entry->word==word) return &entry->value;

        // Iterate to the end of the chain
        Entry* entryIter;
        for (entryIter=entry;entryIter->next!=nullptr;entryIter=entryIter->next) if (entryIter->next->hashValue==hashValue&&entryIter->next->word==word) return &entryIter->next->value;

        // Word is not present, insert and return reference for backpatching
        ++size_;
        entryIter->next=entryAllocator.allocate();
        new (entryIter->next) Entry(word,hashValue);
        return &entryIter->next->value;
    }
}

/// Inserts an element into the hashtable. Returns the value for that key, regardless if an insertion happened or not.
template<class T,class Value,class HashFunc> 
template<class T2>
Value* HashMap<T,Value,HashFunc>::tryAltInsert(const T2& word,uint64_t hashValue) {
    if(capacity() <= (size_t)(size() * LoadFactor)){
        grow();
    }
    Entry*& entry=entries[hashValue&mask];
    // Unroll empty slot branch
    if (entry==nullptr) {
        ++size_;
        entry=entryAllocator.allocate();
        new (entry) Entry(word,hashValue);
        return &entry->value;
    } else {
        // Unroll immediate hit branch
        if (entry->hashValue==hashValue&&entry->word==word) return &entry->value;

        // Iterate to the end of the chain
        Entry* entryIter;
        for (entryIter=entry;entryIter->next!=nullptr;entryIter=entryIter->next) if (entryIter->next->hashValue==hashValue&&entryIter->next->word==word) return &entryIter->next->value;

        // Word is not present, insert and return reference for backpatching
        ++size_;
        entryIter->next=entryAllocator.allocate();
        new (entryIter->next) Entry(word,hashValue);
        return &entryIter->next->value;
    }
}

/// Inserts an element into the hashtable. Returns the value for that key, regardless if an insertion happened or not.
//Only call this method if you are sure, that the value is not already present
template<class T,class Value,class HashFunc>
Value* HashMap<T,Value,HashFunc>::forceInsert(const T& word,uint64_t hashValue) {
    if(capacity() <= (size_t)(size() * LoadFactor)){
        grow();
    }
    Entry*& entry=entries[hashValue&mask];
    // Unroll empty slot branch
    if (entry==nullptr) {
        ++size_;
        entry=entryAllocator.allocate();
        new (entry) Entry(word,hashValue);
        return &entry->value;
    } else {
        ++size_;
        Entry* newEntry=entryAllocator.allocate();
        new (newEntry) Entry(word,hashValue,entry);
        entry=newEntry;
        return &newEntry->value;
    }
}


/// Finds an element inside the hashmap, returns the value if found or nullptr otherwise. Hash value supplied externally
template<class T,class Value,class HashFunc>
Value* HashMap<T,Value,HashFunc>::find(const T& word,uint64_t hashValue) {
    Entry* entry=entries[hashValue&mask];
    // Unroll empty slot branch
    if (entry==nullptr) {
        return nullptr;
    } else {
        // Unroll immediate hit branch
        if (entry->hashValue==hashValue&&entry->word==word) return &entry->value;

        // Iterate to the end of the chain
        for (;entry->next!=nullptr;entry=entry->next) if (entry->next->hashValue==hashValue&&entry->next->word==word) return &entry->next->value;

        return nullptr;
    }
}

/// Alternative find: uses a type that can differ from T, e.g. T=string, T2=string_ref
template<class T,class Value,class HashFunc>
template<class T2>
Value* HashMap<T,Value,HashFunc>::altFind(const T2& word,uint64_t hashValue) {
    Entry* entry=entries[hashValue&mask];
    ++lookupCounter;
    // Unroll empty slot branch
    if (entry==nullptr) {
        return nullptr;
    } else {
        // Unroll immediate hit branch
        if (entry->hashValue==hashValue&&entry->word==word) return &entry->value;

        // Iterate to the end of the chain
        ++collisionCounter;
        for (;entry->next!=nullptr;entry=entry->next,++collisionCounter) if (entry->next->hashValue==hashValue&&entry->next->word==word) return &entry->next->value;

        return nullptr;
    }
}


/// Alternative find which exposes key and value
template<class T,class Value,class HashFunc>
template<class T2>
typename HashMap<T,Value,HashFunc>::Entry* HashMap<T,Value,HashFunc>::altFindEntry(const T2& word, uint64_t hashValue) {
    if(size_ == 0){
        return nullptr;
    }

    Entry* entry=entries[hashValue&mask];
    // Unroll empty slot branch
    if (entry==nullptr) {
        return nullptr;
    } else {
        // Unroll immediate hit branch
        if (entry->hashValue==hashValue&&entry->word==word) return entry;

        // Iterate to the end of the chain
        for (;entry->next!=nullptr;entry=entry->next) if (entry->next->hashValue==hashValue&&entry->next->word==word) return entry->next;

        return nullptr;
    }
}

/// Issue prefetch instruction to directory position of given hashValue
template<class T,class Value,class HashFunc>
void HashMap<T,Value,HashFunc>::prefetchDir(uint64_t hashValue) {
  __builtin_prefetch(&entries[hashValue&mask],0,1);
}

/// Issues prefetch instruction to first entry
template<class T,class Value,class HashFunc>
void HashMap<T,Value,HashFunc>::prefetchFirst(uint64_t hashValue) {
  __builtin_prefetch(entries[hashValue&mask],0,1);
}

/// Pre allocates the hash to be able to contain size elements efficiently. Also, see summary and HashSet
template<class T,class Value,class HashFunc>
void HashMap<T,Value,HashFunc>::hintSize(uint64_t size) {
    this->allocSize=size;
    for (exp=1;(uint64_t)(1<<exp)<size;++exp);
    entries.resize(1<<exp);
    mask=(1ull<<exp)-1ull;
    size_=0;
}

/// Clears all entries in the map
template<class T,class Value,class HashFunc>
void HashMap<T,Value,HashFunc>::clear() {
    memset(entries.data(),0,entries.size()*sizeof(Entry*));
    entryAllocator.reset();
    size_=0;
}
}
