import java.util.ArrayList;
import java.util.List;

// Implementing hashset using linear chaining
public class MyHashTable<T> {
    private static final float maxLoadFactor = 0.75f;
    private int numBuckets; // initial number of buckets; usually a power of 2
    private int size = 0; // number of elements currently saved in hashtable
    private ArrayList<ArrayList<T>> buckets;

    MyHashTable() {
        this(2);
    }

    MyHashTable(int numBuckets) {
        this.numBuckets = numBuckets;
        this.buckets = new ArrayList<>(numBuckets); // (optional) specify initial capacity for no reallocations
        for (int i = 0; i < numBuckets; i++) { // fill array with empty lists
            this.buckets.add(new ArrayList<>());
        }
    }

    public void add(T value) {
        // Add to appropriate bucket
        int hash = calculateBucketOffset(value);
        List<T> bucket = buckets.get(hash);
        for (T key : bucket) {
            if (key == value) { // Avoid duplicates
                return;
            }
        }
        bucket.add(value);
        size++;

        // Rehash ?
        if ((1.0f * size) / numBuckets >= maxLoadFactor) {
            rehash(numBuckets * 2);
        }
    }

    public void remove(T value) {
        int hash = calculateBucketOffset(value);
        List<T> bucket = buckets.get(hash);

        for (int i = 0; i < bucket.size(); i++) {
            if (bucket.get(i) == value) {
                size--;
                bucket.remove(i);
                return;
            }
        }
    }

    public void dump() {
        System.out.println("numBuckets: " + numBuckets);
        System.out.println("size: " + size);
        for (List<T> bucket : buckets) {
            for (T value : bucket) {
                System.out.println(value);
            }
        }
    }

    private int calculateBucketOffset(T value) {
        return Math.abs(value.hashCode() % numBuckets); // save hash, absolute value for negative hashCode
    }

    private void rehash(int newNumBuckets) {
        MyHashTable<T> largerHashTable = new MyHashTable<>(newNumBuckets);
        for (List<T> bucket : buckets) {
            for (T value : bucket) {
                largerHashTable.add(value);
            }
        }
        numBuckets = newNumBuckets;
        buckets = largerHashTable.buckets;
    }

    public static void main(String[] args) {
        MyHashTable<String> test = new MyHashTable<>();

        String[] array = {"What", "does", "this", "even", "mean", "to", "hash", "a", "string", "?", "It", "worked", "maybe", "maybe"};
        for (String s : array) {
            test.add(s);
        }

        test.dump();

        test.remove("What");
        test.remove("hash");

        System.out.println("---");
        test.dump();
    }
}