#pragma once

#include "campers/concurrentmegahashtable.hpp"
#include "util/basengramer.hpp"
#include "util/hashtree.hpp"
#include "util/output.hpp"
#include "util/smallstring.hpp"
#include "util/string.hpp"
#include "util/stringarena.hpp"
#include <string>
#include <unordered_set>
#include <vector>
#include "tbb/enumerable_thread_specific.h"
#include <atomic>


class InterQAwesomeNgramer : public MTBaseNgramer {

   struct MVCCEntry {
      BaseNgramer::SequenceId startVersion;
      BaseNgramer::SequenceId endVersion;
      MVCCEntry* next;

      static const BaseNgramer::SequenceId INIT_START_VERSION = std::numeric_limits<BaseNgramer::SequenceId>::max();
      static const BaseNgramer::SequenceId INIT_END_VERSION = std::numeric_limits<BaseNgramer::SequenceId>::max();

      MVCCEntry()
         : startVersion(INIT_START_VERSION),
           endVersion(INIT_END_VERSION),
           next(nullptr)
      {
      }
   };

   struct AwesomePatternInfo {
      MVCCEntry history;
      static const BaseNgramer::SequenceId INIT_ID = std::numeric_limits<uint32_t>::max();
      uint32_t id;
      uint16_t shortestSuffix;
      uint16_t longestSuffix;

      AwesomePatternInfo() : history(), id(INIT_ID), shortestSuffix(std::numeric_limits<uint16_t>::max()), longestSuffix(0) {}
      //XXX set patternIds after all adds were made in finalize adds

      void setVisibleVersion(BaseNgramer::SequenceId version)
      {
         MVCCEntry* e = &history;
         BaseNgramer::SequenceId startExpected=MVCCEntry::INIT_START_VERSION;
         //if startVersion == max, set it to version
         if(!((std::atomic<BaseNgramer::SequenceId>*)&e->startVersion)->compare_exchange_strong(startExpected,version)){

             //otherwise, create an entry in the chain
             MVCCEntry* n = new MVCCEntry();
             MVCCEntry* expected=nullptr;
             n->startVersion=version;
             while(!((std::atomic<MVCCEntry*>*)&e->next)->compare_exchange_strong(expected,n)){
                 e=expected;
                 expected=nullptr;
             }
         }
         //check if a remove has happened before the pattern was added

         auto eV = e->endVersion;
         if(e->startVersion > eV) {
            ((std::atomic<BaseNgramer::SequenceId>*)&e->endVersion)->compare_exchange_strong(eV,MVCCEntry::INIT_END_VERSION);
         }

         //Case a) start & end == INIT -> default constructed
         //Case b) only start: INIT -> remove has already happened --> if addVersion < endVersion : set addVersion     else: set addVersion && set endVersion to MAX
         //Case c) add entry in chain
      };


      bool isVisible(BaseNgramer::SequenceId version) {

        for(auto h = &history;h;h=h->next)
          if (h->startVersion < version && h->endVersion > version)
            return true;
        return false;
      };


      ///set endVersion to version
      void invalidate(uint32_t version) {
          for(MVCCEntry* e = &history;e;e=e->next)
             if(e->startVersion<version || e->startVersion==MVCCEntry::INIT_START_VERSION){
                 BaseNgramer::SequenceId eV=e->endVersion;
                 while(eV>version && !((std::atomic<BaseNgramer::SequenceId>*)&e->endVersion)->compare_exchange_strong(eV,version))
                     ;
             }
      }
   };

   using ht_t = campers::ConcurrentMegaHashMap<SmallString<>, AwesomePatternInfo, campers::CampersHash64>;
   ht_t patterns;

   using HashType = decltype(patterns)::HashType;

   struct UpdateEntry {
      BorrowedString word;
      HashType hash;
      BaseNgramer::SequenceId version;
      uint16_t length;
      uint16_t wordIndex;

      UpdateEntry(BorrowedString word, HashType hash, BaseNgramer::SequenceId version, uint16_t length, uint16_t wordIndex)
         : word(word), hash(hash), version(version), length(length), wordIndex(wordIndex) {
      }
   };

   struct alignas(64) ThreadState {
      StringArena arena;
      Pool<ht_t::Entry> entryPool;
      std::vector<const char*> words;
      std::vector<HashType> wordHashes;

      ThreadState() : arena(2<<16), words(), wordHashes() {
      }
   };

   struct alignas(64) ThreadQueryState {
      std::vector<const char*> words;
      std::vector<HashType> wordHashes;
      std::vector<bool> seenPatterns;
      std::vector<uint32_t> foundPatterns;

      ThreadQueryState() : words(), wordHashes(), seenPatterns(), foundPatterns() {}
   };



   std::vector<ThreadState> threadStates;
   std::vector<ThreadQueryState> threadQueryStates;
   uint32_t numThreads;





   void calculateWordOffsetsAndHashes(std::vector<const char*>& words, std::vector<HashType>& wordHashes, const BorrowedString& document);

   public:
   InterQAwesomeNgramer() : patterns(2 << 17), threadStates(), threadQueryStates() {}

   void initThreadState(size_t numThreads) override;
   void query(const SequenceId id, const BorrowedString& document, QueryOut& out);
   void intraQuery(const SequenceId id, const BorrowedString& document, QueryOut& out);
   void intraQueryRange(const SequenceId queryVersion, const BorrowedString& document, QueryOut& queryOut, unsigned partition);
   void add(const SequenceId id, const BorrowedString& ngram, const uint32_t patternId){
       size_t adds = 0;
       add(id,ngram, adds, patternId);
       accountAdds(adds);
   };
   void add(const SequenceId id, const BorrowedString& ngram, size_t& adds, const uint32_t patternId);
   void accountAdds(size_t adds);
   void remove(const SequenceId id, const BorrowedString& ngram){
       size_t adds = 0;
       remove(id,ngram, adds);
       accountAdds(adds);
   };
   void remove(const SequenceId id, const BorrowedString& ngram, size_t& adds);
   void finalizeLoading() override;
   inline void finalizeAdds() {};
   inline void checkAndMaybeDoGrow() override {
      patterns.growIfRequired();
   }
};
