#include "baseline/ngramer.hpp"
#include "multithreaded/awesomemt.hpp"
#include "multithreaded/awesomelockfree.hpp"
#include "multithreaded/interquery.hpp"
#include "singlethreaded/awesomengramer.hpp"
#include "singlethreaded/hashngramer.hpp"
#include "singlethreaded/hashtreengramer.hpp"
#include "singlethreaded/hashvecngramer.hpp"
#include "singlethreaded/janngramer.hpp"
#include "singlethreaded/lenvechashvecngramer.hpp"
#include "singlethreaded/ngramer.hpp"
#include "singlethreaded/nhashtreengramer.hpp"
#include "singlethreaded/vechashvecngramer.hpp"
#include "util/input.hpp"
#include "util/log.hpp"
#include "util/output.hpp"
#include "util/sectiontimer.hpp"
#include "util/shared.hpp"
#include "util/statngramer.hpp"
#include "util/string.hpp"
#include "util/workerpool.hpp"
#include "util/prioritypool.hpp"
#include "util/workloadstats.hpp"
#include <iostream>
#include <type_traits>
#include "PerfCall.hpp"

#define PARALLEL_PARSE 0
#define MULTI_PUSH 1

template <typename NgramProcessor>
void runOperations(NgramProcessor& ngramer, std::vector<std::pair<Operation, BorrowedString>>& operations, std::vector<char>& output)
{
   for (auto& operation : operations) {
      switch (operation.first) {
         case Operation::Query: {
            QueryOut out(output);
            ngramer.query(operation.second, out);
            break;
         }
         case Operation::Addition: {
            ngramer.add(operation.second);
            break;
         }
         case Operation::Removal: {
            ngramer.remove(operation.second);
            break;
         }
      }
   }
}

template <typename NgramProcessor> void process()
{
   util::timer::initStartTime();


   std::vector<std::pair<Operation, BorrowedString>> operations;
   BufferedInput input(256ull * 1024ull * 1024ull);
   std::vector<char> output;
   output.reserve(100 * 1024 * 1024);

   SectionTimer timer;
   WorkloadStats workloadStats;
   NgramProcessor rawNgramer;
   auto ngramer = StatNgramer<false, true, decltype(rawNgramer)>(rawNgramer, timer, workloadStats);

   // Untimed loading phase
   {
      // Read initial ngrams
      for (BorrowedString line; input.getline(line) && line != 'S';) {
         operations.emplace_back(std::make_pair(Operation::Addition, line));
      }
      timer.traceParse();

      std::sort(operations.begin(), operations.end(),
                [](const std::pair<Operation, BorrowedString>& a, const std::pair<Operation, BorrowedString>& b) {
                   return (a.second < b.second) * -1;
                });

      runOperations(ngramer, operations, output);
      operations.clear();
      input.reset();

      rawNgramer.finalizeLoading();
      timer.traceAddition();

      workloadStats.finishLoad();
      timer.print();
      timer.reset();
   }

   PerfCall perf;
   if(getenv("PERF")) perf.startProfile("workload");
   std::cout << "R" << std::endl;

   // Timed operation processing
   {
      for (BorrowedString line; input.getline(line);) {
         if (line == 'F') {
            workloadStats.analyzeFlush();
            timer.traceParse();
            runOperations(ngramer, operations, output);
            writeToStdout(output);
            input.reset();
            output.clear();
            operations.clear();
            timer.traceWrite();
         }
         else {
            auto operation = toOperation(line[0]);
            auto parameter = line.substr(2);
            operations.emplace_back(std::make_pair(operation, std::move(parameter)));
         }
      }
   }

   if(getenv("PERF")) perf.endProfile();
   timer.print();
   workloadStats.print();

   assert(operations.empty());
}

template <typename NgramProcessor>
void runInitialAdds(WorkerPool& pool,
                     NgramProcessor& ngramer,
                     const std::vector<std::pair<BaseNgramer::SequenceId, BorrowedString>>& addOperations
                     )
{

  auto& patternIdCounter = ngramer.processor.patternIdCounter;
   parallel_for(pool, 0, addOperations.size(), [&](size_t i) {
      auto operation = addOperations[i];
      ngramer.add(operation.first, operation.second, (uint32_t) (patternIdCounter+i));
   });
   patternIdCounter+=addOperations.size();
   //parallel_each_thread(pool, [&ngramer](size_t){
      ngramer.processor.finalizeAdds();
   //});
   ngramer.timer.traceAddition();
}

template <typename NgramProcessor>
inline void parse(BufferedInput& input, PriorityWorkerPool<NgramProcessor>& pPool,
           BaseNgramer::SequenceId& sequenceId,
           uint32_t& patternIdCounter,
           uint32_t& queryCounter
    ){

    input.eachLine([&](BorrowedString& line){

            if (line == 'F') {
                pPool.writeResults();
                //input.reset();
                queryCounter=0;
            }
            else {
                auto operation = toOperation(line[0]);
                auto parameter = line.substr(2);
                switch (operation) {
                  case Operation::Addition: {
                    pPool.scheduleAdd(PoolOperation(sequenceId++, std::move(parameter), pPool.opCounter++, patternIdCounter++));
                    break;
                  }
                  case Operation::Removal: {
                    pPool.scheduleRemove(PoolOperation(sequenceId++, std::move(parameter), pPool.opCounter++, pPool.REMOVE_MARKER));
                    break;
                  }
                  case Operation::Query: {
                    pPool.scheduleQuery(PoolOperation(sequenceId++, std::move(parameter), pPool.opCounter, queryCounter++));
                    break;
                  }
                }
            }
        });
}

template <typename NgramProcessor> void processMT()
{
   size_t nT;
   if(getenv("THREADS")) nT = atoi(getenv("THREADS"));
   else nT = std::min(10u,std::thread::hardware_concurrency()-2);

   const size_t numThreads = nT;
   std::cerr << "Using " << numThreads << " threads" << std::endl;
   util::timer::initStartTime();

   // Pin main thread before allocating data
   workerPool::pinThread(0);

   BufferedInput input(3ull* 1024ull * 1024ull * 1024ull);

   SectionTimer timer;
   WorkloadStats workloadStats;
   NgramProcessor rawNgramer;
   rawNgramer.initThreadState(numThreads);
   auto ngramer = MTStatNgramer<false, false, decltype(rawNgramer)>(rawNgramer, timer, workloadStats);

   // Untimed loading phase
   {
      WorkerPool workers(0);
      // Read initial ngrams
      std::vector<std::pair<BaseNgramer::SequenceId, BorrowedString>> addOperations;
      for (BorrowedString line; input.getline(line) && line != 'S';) {
         addOperations.emplace_back(std::make_pair(0, line));
      }
      timer.traceParse();

      std::sort(addOperations.begin(), addOperations.end(),
                [](const std::pair<BaseNgramer::SequenceId, BorrowedString>& a,
                   const std::pair<BaseNgramer::SequenceId, BorrowedString>& b) { return (a.second < b.second) * -1; });


      runInitialAdds(workers, ngramer, addOperations);
      addOperations.clear();
      input.reset();

      rawNgramer.finalizeLoading();
      timer.traceAddition();

      workloadStats.finishLoad();
      timer.print();
      timer.reset();
   }

   PriorityWorkerPool<NgramProcessor> pPool(numThreads-1, rawNgramer);

   PerfCall perf;
   if(getenv("PERF")){
       std::string perfMode = "record";
       if(getenv("PERFMODE")) perfMode = getenv("PERFMODE");
       perf.startProfile("workload", perfMode);
   }
   //usleep(1000*1000*1);
   std::cout << "R" << std::endl;

   // Timed operation processing
   BaseNgramer::SequenceId sequenceId = 1;
   {
      uint32_t queryCounter=0;
      auto& patternIdCounter = ngramer.processor.patternIdCounter;
#if PARALLEL_PARSE
      auto p = [&](){
          workerPool::pin(workerPool::nextHyperThreadOf(numThreads),numThreads+1);
          //workerPool::pinThread(numThreads+1);
        parse( input,  pPool,
               sequenceId,
               patternIdCounter,
               queryCounter
          );
      };
      std::thread second(p);

      parse( input,  pPool,
             sequenceId,
             patternIdCounter,
             queryCounter
          );
      second.join();
#else
#if MULTI_PUSH
      const size_t capacity = 16;
      PoolOperation cachedOperations[capacity];
      size_t size = 0;

      auto makeCapacity = [&pPool,&cachedOperations,&size,&queryCounter,&rawNgramer]() {
         if(size>0) {
            pPool.scheduleAddOrRemoveBatched(cachedOperations, size);
         }
         size=0;
         pPool.writeResults();
         rawNgramer.checkAndMaybeDoGrow();
         queryCounter=0;
      };
      auto ensureCapacityQuery = [&pPool,makeCapacity](size_t added) {
         if(!pPool.queryOperations.hasCapacity(added)) {
            makeCapacity();
         }
      };
      auto ensureCapacityAddRemove = [&pPool,makeCapacity,capacity](size_t added) {
         if(!pPool.addsAndRemoves.hasCapacity(added+capacity+1)) {
            makeCapacity();
         }
      };

      for (BorrowedString line; input.getline(line);) {
         if (line == 'F') {
           // Push remaining operations
           pPool.scheduleAddOrRemoveBatched(cachedOperations, size);
           size = 0;

           workloadStats.analyzeFlush();
           timer.traceParse();
           pPool.writeResults();
           rawNgramer.checkAndMaybeDoGrow();
           input.reset();
           queryCounter=0;
           timer.traceWrite();
         }
         else {
             auto operation = toOperation(line[0]);
             auto parameter = line.substr(2);

             switch (operation) {
                 case Operation::Addition: {
                     cachedOperations[size++] = PoolOperation(sequenceId++, std::move(parameter), pPool.opCounter++, patternIdCounter++);
                     break;
                 }
                 case Operation::Removal: {
                     cachedOperations[size++] = PoolOperation(sequenceId++, std::move(parameter), pPool.opCounter++, pPool.REMOVE_MARKER);
                     break;
                 }
                 case Operation::Query: {
                     ensureCapacityQuery(1);
                     pPool.scheduleQuery(PoolOperation(sequenceId++, std::move(parameter), pPool.opCounter, queryCounter++));
                     break;
                 }
             }
             if (size == capacity) {
                 ensureCapacityAddRemove(size);
                 pPool.scheduleAddOrRemoveBatched(cachedOperations, size);
                 size = 0;
             }
         }
      }
#else
        for (BorrowedString line; input.getline(line);) {
            if (line == 'F') {
                workloadStats.analyzeFlush();
                timer.traceParse();
                pPool.writeResults();
                input.reset();
                queryCounter=0;
                timer.traceWrite();
            }
            else {
                auto operation = toOperation(line[0]);
                auto parameter = line.substr(2);
                switch (operation) {
                    case Operation::Addition: {
                        pPool.scheduleAdd(sequenceId++, std::move(parameter), patternIdCounter++);
                        break;
                    }
                    case Operation::Removal: {
                        pPool.scheduleRemove(sequenceId++, std::move(parameter));
                        break;
                    }
                    case Operation::Query: {
                        pPool.scheduleQuery(sequenceId++, std::move(parameter), queryCounter++);
                        break;
                    }
                }
            }
        }
#endif
#endif
   }

   if(getenv("PERF")) perf.endProfile();
   timer.print();
   workloadStats.print();

}

std::map<std::string, std::function<void()>> getAlgorithms()
{
   std::map<std::string, std::function<void()>> variants;
   variants.insert({"best", []() { processMT<InterQAwesomeNgramer>(); }});
   variants.insert({"hash", []() { process<STHashNgramer>(); }});
   variants.insert({"hashtree", []() { process<STHashTreeNgramer>(); }});
   variants.insert({"nhashtree", []() { process<STNHashTreeNgramer>(); }});
   variants.insert({"hashvec", []() { process<STHashVecNgramer>(); }});
   variants.insert({"lenvechashvec", []() { process<STLenVecHashVecNgramer>(); }});
   variants.insert({"vechashvec", []() { process<STVecHashVecNgramer>(); }});
   variants.insert({"jan", []() { process<STJanNgramer>(); }});
   variants.insert({"awesome", []() { process<STAwesomeNgramer>(); }});
//   variants.insert({"awesomemt", []() { processMT<MTAwesomeNgramer>(); }});
//   variants.insert({"awesomelockfree", []() { processMT<MTAwesomeLockfreeNgramer>(); }});
   variants.insert({"interq", []() { processMT<InterQAwesomeNgramer>(); }});
   // variants.insert({"baseline", [](){process<STNgramer>();}});

   return variants;
}

int main(int argc, char** argv)
{
   auto variants = getAlgorithms();
   if (argc != 2) {
      for (auto& variant : variants) {
         std::cout << "" << variant.first << "" << std::endl;
      }
   }
   else {
      std::string variant(argv[1]);
      getAlgorithms()[variant]();
   }

   return 0;
}
