/*
Copyright 2013 Henrik Mühe and Florian Funke

This file is part of CampersCoreBurner.

CampersCoreBurner is free software: you can redistribute it and/or modify
it under the terms of the GNU Affero General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

CampersCoreBurner is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU Affero General Public License for more details.

You should have received a copy of the GNU Affero General Public License
along with CampersCoreBurner. If not, see <http://www.gnu.org/licenses/>.
*/

/*
COMMENT SUMMARY FOR THIS FILE:

This file contains an implementation of an optimized chaining hashset and hashmap.
Depending on the use case, these hashes perform up to 4x better than std::unordered_map.
*/

#pragma once

#include "campers/hash.hpp"
#include "campers/pool.hpp"
#include <atomic>
#include <cstdint>
#include <iostream>
#include <numeric>
#include <vector>
#include <cstdlib>

namespace campers {


/// A fast, chaining HashMap
template <class T, class Value, class HashFunc = class CampersHash> class ConcurrentMegaHashMap {
   public:
   using HashType = typename HashFunc::HashType;

   private:
   static const size_t LoadFactor = 2;

   class DerefIterator {
      public:
      const T& first;
      Value& second;
   };

   class Iterator {
      std::pair<const T&, Value*> data;

      public:
      Iterator(const T& k, Value* v) : data(std::make_pair(k, v)) {}

      bool operator==(const Iterator& other) const { return data.second == other.data.second; }

      bool operator!=(const Iterator& other) const { return data.second != other.data.second; }

      DerefIterator* operator->() { return ((DerefIterator*)(this)); }
   };

  public:
   /// An entry inside the hash map
   struct Entry {
      /// The key
      T word;
      /// The value for this key
      Value value;
      /// The hash value
      HashType hashValue;
      /// Pointer to chain of entries for collisions
      Entry* next;

      /// Constructor
      Entry() : hashValue(0), next(nullptr) {}
      /// Constructor
      Entry(const T& word, HashType hashValue) : word(word), hashValue(hashValue), next(nullptr) {}
   };

  private:
   /// An instance of the hash funtion functor
   HashFunc hashFunc;
   /// The hashmap of pointers to actual entries
   Entry** entries=nullptr;
   uint64_t cap = 0;
   static Pool<Entry>& getPool()
   {
      static Pool<Entry> entry;
      return entry;
   }
   /// The pool allocator.
   Pool<Entry>& entryAllocator;
   /// The exponent, see HashSet and summary for details
   uint64_t exp;
   /// The mask, see HashSet and summary for details
   uint64_t mask;
   /// The number of elements inside the hash,
   std::atomic<size_t> size_;
   /// The size this hash map was allocated with
   uint64_t allocSize;

   public:
   /// Constructor
   ConcurrentMegaHashMap(size_t size) : entryAllocator(getPool()) { hintSize(size); }
   /// Inserts an element into the hashtable. Returns the value for that key, regardless if an insertion happened or
   /// not. Hash value supplied externally
   Value* tryInsert(const T& word, HashType hashValue);
   /// Inserts an element into the hashtable. Returns the value for that key, regardless if an insertion happened or
   /// not.
   Value* tryInsert(const T& word) { return tryInsert(word, hashFunc(word)); }
   /// Finds an element inside the hashmap, returns the value if found or nullptr otherwise. Hash value supplied
   /// externally
   /// Inserts an element into the hashtable. Returns the value for that key, regardless if an insertion happened or
   /// not. Hash value supplied externally
   Value* tryInsertLocked(const T& word, HashType hashValue);
   /// Inserts an element into the hashtable. Returns the value for that key, regardless if an insertion happened or
   /// not. Hash value supplied externally. Optimistic insert
   Value* tryInsertLockfree(const T& word, HashType hashValue,Pool<Entry>& threadPrivatePool,uint64_t& localAddCounter);

   Iterator find(const T& word, HashType hashValue) const;
   /// Finds an element inside the hashmap, returns the value if found or nullptr otherwise
   Iterator find(const T& word) const
   {
      if (size_ == 0) {
         return end();
      }
      else {
         return find(word, hashFunc(word));
      }
   }

   void erase(const T& word, HashType hashValue);
   void erase(const T& word)
   {
      if (size_ == 0) {
         return;
      }
      else {
         return erase(word, hashFunc(word));
      }
   }

   void growIfRequired() {
      if (capacity() <= (size_t)(size() * LoadFactor)) {
         grow(4 * capacity());
      }
   }

   /// Alternative find: uses a type that can differ from T, e.g. T=string, T2=string_ref
   template <class T2> inline Value* altFind(const T2& word) { return altFind(word, hashFunc(word)); }
   /// Alternative find: uses a type that can differ from T, e.g. T=string, T2=string_ref
   template <class T2> inline Value* altFind(const T2& word, HashType hashValue);
   /// Pre allocates the hash to be able to contain size elements efficiently. Also, see summary and HashSet
   void hintSize(size_t size);
   /// Returns the current number of elements inside the hash
   size_t size() const { return size_; }
   /// atomically adds s to size
   void addToSize(size_t s){size_+=s;};
   /// Grows the hash to twice its capacity
   void grow() { grow(2 * capacity()); };
   /// Grows the hash to be able to efficiently contain 2* capacity elements
   void grow(size_t newSize);
   /// Clears all entries in the HashMap
   // void clear();
   /// Returns the current capacity of the hash
   size_t capacity() const { return cap; };

   void prefetchBucketPtr(uint64_t hashValue) const { __builtin_prefetch(&entries[hashValue & mask], 0, 1); }
   void prefetchBucketEntry(uint64_t hashValue) const { __builtin_prefetch(entries[hashValue & mask], 0, 1); }


   // XXX: Custom code
   /// TODO: Disallow after migraiton
   ConcurrentMegaHashMap() : ConcurrentMegaHashMap(32){};
   HashType hash(const T& word) const { return hashFunc(word); }
   ConcurrentMegaHashMap& operator=(const ConcurrentMegaHashMap&) = delete;
   ConcurrentMegaHashMap(const ConcurrentMegaHashMap& other) = delete;

   ConcurrentMegaHashMap(ConcurrentMegaHashMap&& other)
      : hashFunc(std::move(other.hashFunc)), entries(other.entries), entryAllocator(other.entryAllocator)
   {
      exp = other.exp;
      cap = other.cap;
      mask = other.mask;
      size_ = other.size_;
      allocSize = other.allocSize;
      other.entries=nullptr;
   }

   Iterator end() const { return Iterator(*((T*)nullptr), nullptr); }
   /// Returns 1 if the value is contained
   size_t count(const T& word) { return find(word) != nullptr ? 1 : 0; }
   /// Clears all entries in the map
   void clear();

   Value& operator[](const T& word)
   {
      auto match = find(word);
      if (match == end()) {
         auto val = tryInsert(word);
         new (val) Value();
         return *val;
      }
      else {
         return match->second;
      }
   }
};


/// Grows the hash to be able to efficiently contain 2* capacity elements
template <class T, class Value, class HashFunc> void ConcurrentMegaHashMap<T, Value, HashFunc>::grow(size_t newSize)
{
   allocSize = newSize;
   // Grow
   for (exp = 1; (size_t)(1ull << exp) < newSize; ++exp) {
   }
   auto newCap=1<<exp;
   Entry** newEntries=static_cast<Entry**>(std::malloc(newCap*sizeof(Entry*)));
   memset(newEntries, 0, newCap * sizeof(Entry*));
   mask = (1ull << exp) - 1ull;

   // Copy entries over
   for (Entry** entryChain=entries, ** end=entryChain+cap; entryChain<end;++entryChain) {
       for (Entry* e = *entryChain, *next; e != nullptr; e = next) {
         Entry** entry = &newEntries[e->hashValue & mask];
         next = e->next;
         e->next = *entry;
         *entry = e;
      }
   }

   if(entries) free(entries);
   // Switch to new entry vector
   entries = newEntries;
   cap=newCap;
}

template <typename Content> struct ConcLockedPtr {
   std::atomic<Content*>* ref;
   Content* original;

   ConcLockedPtr(std::atomic<Content*>* contentPtrRef) : ref(contentPtrRef)
   {
      while (!acquireLock(*ref)) {
      }
   }

   bool acquireLock(Content* ptr)
   {
      if (!isLocked(ptr) && ref->compare_exchange_weak(ptr, makeLocked(ptr))) {
         original = ptr;
         return true;
      }
      else {
         return false;
      }
   }

   void release() { *ref = removeLock(*ref); }

   static bool isLocked(Content* ptr) { return (((uintptr_t)ptr) & (lockMask)) != 0; }

   static Content* removeLock(Content* ptr) { return (Content*)(((uintptr_t)ptr) & (~lockMask)); }

   static Content* makeLocked(Content* ptr) { return (Content*)(((uintptr_t)ptr) | lockMask); }

   static constexpr uintptr_t lockMask = (1ull << ((sizeof(Content*) * 8) - 1));
};

/// Inserts an element into the hashtable. Returns the value for that key, regardless if an insertion happened or not.
template <class T, class Value, class HashFunc>
Value* ConcurrentMegaHashMap<T, Value, HashFunc>::tryInsertLocked(const T& word, HashType hashValue)
{
    static std::atomic<Value*> allocatePtr;

   //TODO: TK: this seems problematic. In case of growth, some values may be inserted
   // in parallel into the old HT and not be transfered to the new HT
   if (capacity() <= (size_t)(size() * LoadFactor)) {
      grow();
   }

   ConcLockedPtr<Entry> lockedPtr(&entries[hashValue & mask]);

   Entry* entry = lockedPtr.original;
   // Unroll empty slot branch
   if (entry == nullptr) {
      ++size_;
      ConcLockedPtr<Value> allocLock(&allocatePtr);
      auto newEntry = entryAllocator.allocate();
      allocLock.release();
      new (newEntry) Entry(word, hashValue);
      auto value = &newEntry->value;
      entries[hashValue & mask] = newEntry;
      return value;
   }
   else {
      // Unroll immediate hit branch
      if (entry->hashValue == hashValue && entry->word == word) {
         auto value = &entry->value;
         lockedPtr.release();
         return value;
      }

      // Iterate to the end of the chain
      Entry* entryIter;
      for (entryIter = entry; entryIter->next != nullptr; entryIter = entryIter->next){
         Entry* eN = entryIter->next;
         if (eN->hashValue == hashValue && eN->word == word) {
            auto value = &eN->value;
            lockedPtr.release();
            return value;
         }
      }

      // Word is not present, insert and return reference for backpatching
      ++size_;
      ConcLockedPtr<Value> allocLock(&allocatePtr);
      entryIter->next = entryAllocator.allocate();
      allocLock.release();
      new (entryIter->next) Entry(word, hashValue);
      Entry* eN = entryIter->next;
      auto value = &eN->value;
      lockedPtr.release();
      return value;
   }
}


/// Inserts an element into the hashtable. Returns the value for that key, regardless if an insertion happened or not.
template <class T, class Value, class HashFunc>
Value* ConcurrentMegaHashMap<T, Value, HashFunc>::tryInsertLockfree(const T& word, HashType hashValue,Pool<Entry>& threadPrivatePool, uint64_t& localAddCounter)
{
   static std::atomic<Value*> allocatePtr;

   std::atomic<Entry*>* entryPtrLoc = (std::atomic<Entry*>*)&entries[hashValue & mask];
   Entry* entry = *entryPtrLoc;
   Entry* newEntry;
   // Unroll empty slot branch
   if (entry == nullptr) {
      newEntry = new (threadPrivatePool.allocate()) Entry(word, hashValue);
      if(!entryPtrLoc->compare_exchange_strong(entry, newEntry)){
          // another thread changed "entry", thus handle the chain
          goto chainCase;
      }
      //++size_;
      ++localAddCounter;
      return &newEntry->value;
   }
   else {
      newEntry = new (threadPrivatePool.allocate()) Entry(word, hashValue);
   chainCase:
      do{
         // Unroll immediate hit branch
         if (entry->hashValue == hashValue && entry->word == word)
             return &entry->value;

         // Iterate to the end of the chain
         for (; entry->next != nullptr; entry = entry->next){
            Entry* eN = entry->next;
            if (eN->hashValue == hashValue && eN->word == word)
                return &eN->value;
         }

         // Word is not present, insert and return reference for backpatching
         entryPtrLoc = (std::atomic<Entry*>*)(&entry->next);
         entry=nullptr;
      }while(!entryPtrLoc->compare_exchange_strong(entry, newEntry));

      //++size_;
      ++localAddCounter;
      return &newEntry->value;
   }
}

/// Inserts an element into the hashtable. Returns the value for that key, regardless if an insertion happened or not.
template <class T, class Value, class HashFunc>
Value* ConcurrentMegaHashMap<T, Value, HashFunc>::tryInsert(const T& word, HashType hashValue)
{
   if (capacity() <= (size_t)(size() * LoadFactor)) {
      grow();
   }

   Entry*& entry = entries[hashValue & mask];
   // Unroll empty slot branch
   if (entry == nullptr) {
      ++size_;
      entry = entryAllocator.allocate();
      new (entry) Entry(word, hashValue);
      return &entry->value;
   }
   else {
      // Unroll immediate hit branch
      if (entry->hashValue == hashValue && entry->word == word)
         return &entry->value;

      // Iterate to the end of the chain
      Entry* entryIter;
      for (entryIter = entry; entryIter->next != nullptr; entryIter = entryIter->next)
         if (entryIter->next->hashValue == hashValue && entryIter->next->word == word)
            return &entryIter->next->value;

      // Word is not present, insert and return reference for backpatching
      ++size_;
      entryIter->next = entryAllocator.allocate();
      new (entryIter->next) Entry(word, hashValue);
      return &entryIter->next->value;
   }
}

/// Finds an element inside the ConcurrentMegaHashMap, returns the value if found or nullptr otherwise. Hash value supplied
/// externally
template <class T, class Value, class HashFunc>
typename ConcurrentMegaHashMap<T, Value, HashFunc>::Iterator ConcurrentMegaHashMap<T, Value, HashFunc>::find(const T& word, HashType hashValue) const
{
   if (size() == 0) {
      return Iterator(word, nullptr);
   }

   Entry* entry = entries[hashValue & mask];
   // Unroll empty slot branch
   if (entry == nullptr) {
      return Iterator(word, nullptr);
   }
   else {
      // Unroll immediate hit branch
      if (entry->hashValue == hashValue && entry->word == word)
         return Iterator(word, &entry->value);

      // Iterate to the end of the chain
      for (; entry->next != nullptr; entry = entry->next)
         if (entry->next->hashValue == hashValue && entry->next->word == word)
            return Iterator(word, &entry->next->value);

      return Iterator(word, nullptr);
   }
}

template <class T, class Value, class HashFunc>
void ConcurrentMegaHashMap<T, Value, HashFunc>::erase(const T& word, HashType hashValue)
{
   if (size() == 0) {
      return;
   }

   Entry* entry = entries[hashValue & mask];
   // Unroll empty slot branch
   if (entry == nullptr) {
      return;
   }
   else {
      // Unroll immediate hit branch
      if (entry->hashValue == hashValue && entry->word == word) {
         if (entry->next == nullptr) {
            entryAllocator.deallocate(entry);
            entries[hashValue & mask] = nullptr;
         }
         else {
            entry = entry->next;
         }
         return;
      }

      // Iterate to the end of the chain
      for (; entry->next != nullptr; entry = entry->next) {
         if (entry->next->hashValue == hashValue && entry->next->word == word) {
            entry->next = entry->next->next;
            return;
         }
      }
   }
}

/// Alternative find: uses a type that can differ from T, e.g. T=string, T2=string_ref
template <class T, class Value, class HashFunc>
template <class T2>
Value* ConcurrentMegaHashMap<T, Value, HashFunc>::altFind(const T2& word, HashType hashValue)
{
   Entry* entry = entries[hashValue & mask];
   // Unroll empty slot branch
   if (entry == nullptr) {
      return nullptr;
   }
   else {
      // Unroll immediate hit branch
      if (entry->hashValue == hashValue && entry->word == word)
         return &entry->value;

      // Iterate to the end of the chain
      for (; entry->next != nullptr; entry = entry->next){
         Entry* eN = entry->next;
         if (eN->hashValue == hashValue && eN->word == word)
            return &eN->value;
      }

      return nullptr;
   }
}

/// Pre allocates the hash to be able to contain size elements efficiently. Also, see summary and HashSet
template <class T, class Value, class HashFunc> void ConcurrentMegaHashMap<T, Value, HashFunc>::hintSize(size_t size)
{
   this->allocSize = size;
   for (exp = 1; (size_t)(1ull << exp) < size; ++exp)
      ;
   if(entries) free(entries);
   cap=1<<exp;
   entries=static_cast<Entry**>(std::malloc(cap*sizeof(Entry*)));
   memset(entries, 0, cap * sizeof(Entry*));
   mask = (1ull << exp) - 1ull;
   size_ = 0;
}

// XXX: Custom code

/// Clears all entries in the map
template <class T, class Value, class HashFunc> void ConcurrentMegaHashMap<T, Value, HashFunc>::clear()
{
   memset(entries, 0, capacity() * sizeof(Entry*));
   entryAllocator.reset();
   size_ = 0;
}
}
