#include "util/workerpool.hpp"
#include "util/log.hpp"
#include <cassert>
#include <pthread.h>
#include <numa.h>

namespace {
static thread_local size_t localThreadId;
}

static const uint32_t numSockets = numa_num_configured_nodes();
static const auto numThreads=std::thread::hardware_concurrency();
static const uint32_t threadsPerCore = 2; // TODO: find number of hyperthreads per core with API call
static const uint32_t coresPerSocket = numThreads/numSockets/threadsPerCore;
static const uint32_t numCores = coresPerSocket*numSockets;

uint32_t orderCoresSocketsHyperThreads(uint32_t threadId){
    auto threadsPerSocket = (coresPerSocket*threadsPerCore);
    auto thread = threadId/threadsPerSocket%threadsPerCore;
    auto socket = threadId/coresPerSocket%numSockets;
    auto core = threadId%coresPerSocket;
    return socket+core*numSockets+thread*(numCores);
}

uint32_t odd(uint32_t i){return i&1;}
uint32_t orderHyperThreadsCoresSockets(uint32_t threadId){
    auto threadsPerSocket = (coresPerSocket*threadsPerCore);
    auto socket = threadId/threadsPerSocket%numSockets;
    auto core = threadId/threadsPerCore%coresPerSocket;
    auto thread = threadId%threadsPerCore;
    return socket+core*numSockets+thread*(numCores);
}

uint32_t orderCoresHyperThreadsSockets(uint32_t threadId){
    auto threadsPerSocket = (coresPerSocket*threadsPerCore);
    auto socket = threadId/threadsPerSocket%numSockets;
    auto thread = threadId/coresPerSocket%threadsPerCore;
    auto core = threadId%coresPerSocket;
    return socket+core*numSockets+thread*(numCores);
}

uint32_t orderSocketsCoresHyperThreads(uint32_t threadId){
    return threadId%numThreads;
}

uint32_t workerPool::nextHyperThreadOf(uint32_t threadId){
    auto threadsPerSocket = (coresPerSocket*threadsPerCore);
    auto socket = threadId/threadsPerSocket%numSockets;
    auto thread = (threadId/coresPerSocket+1)%threadsPerCore;
    auto core = threadId%coresPerSocket;
    return socket+core*numSockets+thread*(numCores);
}




void workerPool::pin(size_t hwThreadId, size_t threadId){
    localThreadId=threadId;
    pthread_t currentThread = pthread_self();
    pthread_setname_np(currentThread, ("workerPool " + std::to_string(threadId)).c_str());
    cpu_set_t cpuset;
    CPU_ZERO(&cpuset);
    CPU_SET(hwThreadId, &cpuset);
    if (pthread_setaffinity_np(currentThread, sizeof(cpu_set_t), &cpuset) != 0) {
        FATAL_ERROR("Could not pin thread " << threadId << " to thread " << hwThreadId);
    }
}

void workerPool::pinThread(size_t threadId)
{
#if 1
   auto coreId = threadId*2;
   pin(coreId, threadId);
#else 
	auto numCores = std::thread::hardware_concurrency();
	if (threadId >= numCores) {
		FATAL_ERROR("Invalid thread id " << threadId);
	}

	cpu_set_t cpuset;
	CPU_ZERO(&cpuset);
	auto coreId = [threadId,numCores]() {
		if(threadId<(numCores/2)) {
			return threadId*2;
		} else {
			return ((threadId-(numCores/2))*2)+1;
		}
		//TODO Check if this is correct
	}();
	//std::cerr<<coreId<<std::endl;
	CPU_SET(coreId, &cpuset);
	pthread_t currentThread = pthread_self();
	if (pthread_setaffinity_np(currentThread, sizeof(cpu_set_t), &cpuset) != 0) {
		FATAL_ERROR("Could not pin thread " << threadId);
	}

	localThreadId=threadId;
#endif
}

size_t workerPool::getWorkerThreadId() {
   return localThreadId;
}
void workerPool::yield(unsigned k) {
  if (k < 4) {
   } else if (k < 16) {
     __asm__ ( "pause;" );
   } else if (k < 32 || k & 1) {
      sched_yield();
   } else {
      sched_yield();
      /*struct timespec rqtp = { 0, 0 };
      rqtp.tv_sec = 0;
      rqtp.tv_nsec = 1000;
      nanosleep( &rqtp, 0 );*/
   }
}




WorkerPool::WorkerPool(size_t numWorkers)
   : running(true), activeWorkers(0), currentWorkVersion(0), currentWork(nullptr), blocked(false), blockedCond(), blockedMutex()
{
   for (size_t i = 0; i < numWorkers; i++) {
      auto id = i;
      threads.emplace_back([this, id]() {
          workerPool::pinThread(id);
         size_t processedVersion = 0;
         while (running) {
            size_t newWorkVersion = currentWorkVersion.load();
            if (processedVersion != newWorkVersion) {
               assert(currentWork != nullptr);
               currentWork(id);
               activeWorkers.fetch_sub(1);
               processedVersion = newWorkVersion;
            }
            else {
               asm("pause");
               asm("pause");
               asm("pause");
            }
         }
      });
   }
   // Pin main thread also
   workerPool::pinThread(numWorkers);
}

WorkerPool::~WorkerPool()
{
   if(blocked) {
      unblock();
   }
   running = false;
   for (auto& thread : threads) {
      thread.join();
   }
}
void WorkerPool::submit(std::function<void(size_t)> function)
{
   assert(activeWorkers == 0);
   assert(!blocked);
   currentWork = function;
   activeWorkers = threads.size();
   currentWorkVersion.fetch_add(1);
}

void WorkerPool::waitForCompletition()
{
   while (activeWorkers > 0) {
   }
   currentWork = nullptr;
}

void WorkerPool::block()
{
   blocked = true;
   submit([this](size_t) {
      {
         std::unique_lock<std::mutex> lock(blockedMutex);
         while (blocked) { // loop to avoid spurious wakeups
            blockedCond.wait(lock);
         }
      }
      blockedCond.notify_all();
   });
}

void WorkerPool::unblock()
{
   {
      std::unique_lock<std::mutex> lock(blockedMutex);
      blocked = false;
   }
   blockedCond.notify_all();
   waitForCompletition();
}
