#pragma once

#include "util/string.hpp"
#include <cstdint>

template <typename BaseType = char> class SmallString {
   static const unsigned SHORT_STRING_LIMIT = 28;
   static const unsigned PREFIX_LENGTH = 20;

   union {
      struct {
         uint32_t length;
         BaseType data[SHORT_STRING_LIMIT];
      } inlined;
      struct {
         uint32_t length;
         BaseType prefix[PREFIX_LENGTH];
         const char* pointer;
      } full;
   };

   public:
   inline uint32_t size() const { return inlined.length; }
   inline bool isInlined() const { return size() <= SHORT_STRING_LIMIT; }

   SmallString(const BaseType* data, uint32_t length)
   {
      memset(this, 0, sizeof(SmallString));
      inlined.length = length;
      if (isInlined()) {
         memcpy(&inlined.data, data, length);
      }
      else {
         memcpy(&full.prefix, data, PREFIX_LENGTH);
         full.pointer = data;
      }
   };

   SmallString(const BorrowedString str) : SmallString(str.data(), str.size()) {}

   inline bool operator==(const SmallString& other) const
   {
      if (!memcmp(this, &other, sizeof(full.length) + PREFIX_LENGTH)) {
         if (isInlined()) {
            return full.pointer == other.full.pointer;
         }
         else {
            return BorrowedString::matchSSE(full.pointer, size(), other.full.pointer, size());
         }
      }
      return false;
   }

   inline bool operator==(const BorrowedString& other) const
   {
      if (size() != other.size()) {
         return false;
      }

      const char* ptr = [this]() {
         if (isInlined()) {
            return inlined.data;
         }
         else {
            return full.pointer;
         }
      }();

      return BorrowedString::matchSSE(ptr, size(), other.data(), size());
   }

   static inline bool needsAllocation(const BorrowedString& ngram) {
      return ngram.size() >= SHORT_STRING_LIMIT;
   }

   BorrowedString asBorrowed() const
   {
      if (isInlined()) {
         return BorrowedString(inlined.data, inlined.length);
      }
      else {
         return BorrowedString(full.pointer, full.length);
      }
   }
};

namespace std {
template <> struct hash<SmallString<char>> {
   size_t operator()(const SmallString<char>& x) const noexcept { return std::hash<BorrowedString>()(x.asBorrowed()); }
};
}
