#pragma once

#include "campers/megahashtable.hpp"
#include <algorithm>
#include <cassert>
#include <iostream>
#include <unordered_map>
#include <vector>

template <typename KeyType, typename SplitFunc> class HashTree {
   static const uint32_t InitialState = 0;
   static const uint32_t NoMatchState = std::numeric_limits<uint32_t>::max();

   std::vector<campers::MegaHashMap<KeyType, uint32_t>> stateMaps;
   std::vector<bool> terminals;
   std::vector<KeyType> keys;
   uint32_t nextStateIndex = InitialState + 1;
   SplitFunc& split;

   public:
   HashTree(SplitFunc& split) : stateMaps(), terminals(), keys(), nextStateIndex(InitialState + 1), split(split)
   {
      ensureCapacity(nextStateIndex);
      stateMaps[InitialState].grow(2 << 16);
   }

   void optimizeGrowth()
   {
      ensureCapacity(nextStateIndex * 3);
      stateMaps[InitialState].grow();
      for (auto& map : stateMaps) {
         map.grow();
      }
   }

   void add(const KeyType& key)
   {
      uint32_t currentIndex = InitialState;
      auto splits = split(key);
      while (splits.first.size() != 0) {
         const auto& current = stateMaps[currentIndex];
         auto& value = splits.first;
         auto match = current.find(value);
         if (match == current.end()) {
            auto assignedIndex = nextStateIndex++;
            ensureCapacity(assignedIndex);
            stateMaps[currentIndex][value] = assignedIndex;
            currentIndex = assignedIndex;
         }
         else {
            currentIndex = match->second;
         }

         splits = split(splits.second);
      }

      assert(splits.second.size() == 0);
      terminals[currentIndex] = true;
      keys[currentIndex] = key;
   }

   void remove(const KeyType& key)
   {
      uint32_t currentIndex = InitialState;
      auto splits = split(key);
      while (splits.first.size() != 0) {
         const auto& current = stateMaps[currentIndex];
         auto& value = splits.first;
         auto match = current.find(value);
         if (match == current.end()) {
            return;
         }
         else {
            currentIndex = match->second;
         }

         splits = split(splits.second);
      }

      assert(splits.second.size() == 0);
      terminals[currentIndex] = false;
   }

   std::pair<uint32_t, bool> query(const KeyType& key, uint32_t startNode = InitialState) const
   {
      if (key.size() == 0) {
         return std::make_pair(NoMatchState, false);
      }

      uint32_t currentIndex = startNode;
      auto splits = split(key);
      while (splits.first.size() != 0) {
         auto& current = stateMaps[currentIndex];
         auto& value = splits.first;
         auto match = current.find(value);
         if (match == current.end()) {
            return std::make_pair(NoMatchState, false);
         }
         else {
            currentIndex = match->second;
         }

         splits = split(splits.second);
      }

      return std::make_pair(currentIndex, terminals[currentIndex]);
   }

   std::pair<uint32_t, bool> fastQuery(const KeyType& key, uint32_t hash, uint32_t startNode = InitialState) const
   {
      if (key.size() == 0) {
         return std::make_pair(NoMatchState, false);
      }

      auto& current = stateMaps[startNode];
      auto match = current.find(key, hash);
      if (match == current.end()) {
         return std::make_pair(NoMatchState, false);
      }
      else {
         auto currentIndex = match->second;
         return std::make_pair(currentIndex, terminals[currentIndex]);
      }
   }

   const KeyType& getKey(uint32_t node) { return keys[node]; }

   std::pair<uint32_t, bool> end() const { return std::make_pair(NoMatchState, false); }

   void ensureCapacity(size_t desired)
   {
      if (stateMaps.size() <= desired) {
         stateMaps.resize(desired + 1);
         terminals.resize(desired + 1);
         keys.resize(desired + 1);
      }
   }

   class HashTreeIterator {
      const HashTree<KeyType, SplitFunc>& hashtree;
      std::vector<std::pair<uint32_t, uint32_t>> states;
      std::vector<std::pair<uint32_t, uint32_t>> matches;
      uint32_t currentDepth;

      public:
      HashTreeIterator(const HashTree<KeyType, SplitFunc>& hashtree) : hashtree(hashtree), states(), currentDepth(0)
      {
         states.reserve(4);
      }

      void match(const KeyType& key)
      {
         currentDepth++;

         int32_t last = states.size() - 1;
         const auto hash = hashtree.stateMaps[InitialState].hash(key);
         if (states.size() > 0) {
            for (int32_t i = 0; i <= last; i++) {
               auto& current = hashtree.stateMaps[states[i].second];
               auto match = current.find(key, hash);
               if (match != current.end()) {
                  states[i].second = match->second;
               }
               else {
                  states[i].swap(states[last]);
                  last--;
                  i--;
               }
            }
            states.resize(last + 1);
         }

         auto& current = hashtree.stateMaps[InitialState];
         auto match = current.find(key, hash);
         if (match != current.end()) {
            states.emplace_back(std::make_pair(currentDepth, match->second));
         }
      }

      bool isMatch() const
      {
         for (auto& state : states) {
            if (hashtree.terminals[state.second]) {
               return true;
            }
         }

         return false;
      }

      void finish() { states.clear(); }

      template <typename MatchProcessor> void handleMatches(MatchProcessor processor)
      {
         if (states.size() == 0 && matches.size() == 0) {
            return;
         }

         for (auto& state : states) {
            if (hashtree.terminals[state.second]) {
               matches.push_back(std::make_pair(state.first, state.second));
            }
         }

         auto allowedOut = currentDepth + 1000;
         if (matches.size() == 0 || states.size() > 1) {
            return;
         }
         else if (states.size() == 1) {
            allowedOut = states[0].first;
         }

         if (matches.size() == 1) {
            if (matches[0].first < allowedOut) {
               processor(hashtree.keys[matches[0].second]);
               matches.clear();
            }
            return;
         }
         else {
            std::sort(matches.begin(), matches.end());
            matches.erase(std::remove_if(matches.begin(), matches.end(),
                                         [allowedOut, this, processor](std::pair<uint32_t, uint32_t>& match) {
                                            if (match.first < allowedOut) {
                                               processor(hashtree.keys[match.second]);
                                               return true;
                                            }
                                            else {
                                               return false;
                                            }
                                         }),
                          matches.end());
         }
      }
   };

   HashTreeIterator matcher() { return HashTreeIterator(*this); }
};

template <typename KeyType, typename SplitFunc> const uint32_t HashTree<KeyType, SplitFunc>::NoMatchState;
template <typename KeyType, typename SplitFunc> const uint32_t HashTree<KeyType, SplitFunc>::InitialState;
