#pragma once

#include "campers/megahashtable.hpp"
#include "util/basengramer.hpp"
#include "util/hashtree.hpp"
#include "util/output.hpp"
#include "util/smallstring.hpp"
#include "util/string.hpp"
#include "util/stringarena.hpp"
#include <string>
#include <unordered_set>
#include <vector>

class MTAwesomeNgramer : public MTBaseNgramer {
   struct MVCCEntry {
      BaseNgramer::SequenceId startVersion;
      BaseNgramer::SequenceId endVersion;
      BaseNgramer::SequenceId seenVersion;
      MVCCEntry* next;

      MVCCEntry()
         : startVersion(std::numeric_limits<BaseNgramer::SequenceId>::max()),
           endVersion(std::numeric_limits<BaseNgramer::SequenceId>::max()),
           seenVersion(std::numeric_limits<BaseNgramer::SequenceId>::max()), next(nullptr)
      {
      }
   };

   struct AwesomePatternInfo {
      MVCCEntry history;
      uint16_t shortestSuffix;
      uint16_t longestSuffix;

      AwesomePatternInfo() : history(), shortestSuffix(std::numeric_limits<uint16_t>::max()), longestSuffix(0) {}

      void setVisibleVersion(BaseNgramer::SequenceId version)
      {
         MVCCEntry* e = &history;
         BaseNgramer::SequenceId startExpected=std::numeric_limits<BaseNgramer::SequenceId>::max();
         //if startVersion == max, set it to version
         if(!((std::atomic<BaseNgramer::SequenceId>*)&e->startVersion)->compare_exchange_strong(startExpected,version)){

             //otherwise, create an entry in the chain
             MVCCEntry* n = new MVCCEntry();
             MVCCEntry* expected=nullptr;
             n->startVersion=version;
             n->seenVersion=version;
             while(!((std::atomic<MVCCEntry*>*)&e->next)->compare_exchange_strong(expected,n)){
                 e=expected;
                 expected=nullptr;
             }
         }else{
             history.seenVersion = version;
         }
      };

      bool updateIfUnseen(BaseNgramer::SequenceId version)
      {
          if(history.seenVersion < version){
              history.seenVersion = version;
              for(auto h = &history;h;h=h->next)
                  if (h->startVersion < version && h->endVersion > version)
                      return true;
          }
          return false;
      };

      void invalidate(uint32_t version) {
          for(MVCCEntry* e = &history;e;e=e->next)
             if(e->startVersion<version){
                 BaseNgramer::SequenceId eV=e->endVersion;
                 while(eV>version && !((std::atomic<BaseNgramer::SequenceId>*)&e->endVersion)->compare_exchange_strong(eV,version))
                     ;
             }
      }
   };

   campers::MegaHashMap<SmallString<>, AwesomePatternInfo, campers::CampersHash64> patterns;

   using HashType = decltype(patterns)::HashType;

   struct UpdateEntry {
      BorrowedString word;
      HashType hash;
      BaseNgramer::SequenceId version;
      uint16_t length;
      uint16_t wordIndex;

      UpdateEntry(BorrowedString word, HashType hash, BaseNgramer::SequenceId version, uint16_t length, uint16_t wordIndex)
         : word(word), hash(hash), version(version), length(length), wordIndex(wordIndex) {
      }
   };

   struct alignas(64) ThreadState {
      StringArena arena;
      std::vector<const char*> words;
      std::vector<HashType> wordHashes;
      std::vector<UpdateEntry> delayedUpdates;

      ThreadState() : arena(2<<16), words(), wordHashes(),delayedUpdates() {
      }
   };

   std::vector<ThreadState> threadStates;

   public:
   MTAwesomeNgramer() : patterns(2 << 17), threadStates() {}

   void initThreadState(size_t numThreads) override;
   void query(const SequenceId id, const BorrowedString& document, QueryOut& out) override;
   void intraQuery(const SequenceId id, const BorrowedString& document, QueryOut& out) { query(id, document, out); }
   void add(const SequenceId id, const BorrowedString& ngram, const uint32_t patternId);
   void add(const SequenceId id, const BorrowedString& ngram,size_t&, const uint32_t patternId){
       add(id,ngram, patternId);
   }
   void accountAdds(size_t adds);
   void remove(const SequenceId id, const BorrowedString& ngram);
   void remove(const SequenceId id, const BorrowedString& ngram, size_t&){
       remove(id,ngram);
   }
   void finalizeLoading() override;
   void finalizeAdds();
};
