/*
Copyright 2013 Henrik Mühe and Florian Funke

This file is part of CampersCoreBurner.

CampersCoreBurner is free software: you can redistribute it and/or modify
it under the terms of the GNU Affero General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

CampersCoreBurner is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU Affero General Public License for more details.

You should have received a copy of the GNU Affero General Public License
along with CampersCoreBurner. If not, see <http://www.gnu.org/licenses/>.
*/

/*
COMMENT SUMMARY FOR THIS FILE:

This file contains an implementation of an optimized chaining hashset and hashmap.
Depending on the use case, these hashes perform up to 4x better than std::unordered_map.
*/

#pragma once

#include "campers/hash.hpp"
#include "campers/pool.hpp"
#include <atomic>
#include <cstdint>
#include <iostream>
#include <numeric>
#include <vector>

namespace campers {


/// A fast, chaining HashMap
template <class T, class Value, class HashFunc = class CampersHash> class MegaHashMap {
   public:
   using HashType = typename HashFunc::HashType;

   private:
   static const size_t LoadFactor = 2;

   class DerefIterator {
      public:
      const T& first;
      Value& second;
   };

   class Iterator {
      std::pair<const T&, Value*> data;

      public:
      Iterator(const T& k, Value* v) : data(std::make_pair(k, v)) {}

      bool operator==(const Iterator& other) const { return data.second == other.data.second; }

      bool operator!=(const Iterator& other) const { return data.second != other.data.second; }

      DerefIterator* operator->() { return ((DerefIterator*)(this)); }
   };

public:
   /// An entry inside the hash map
   struct Entry {
      /// The key
      T word;
      /// The value for this key
      Value value;
      /// The hash value
      HashType hashValue;
      /// Pointer to chain of entries for collisions
      Entry* next;

      /// Constructor
      Entry() : hashValue(0), next(nullptr) {}
      /// Constructor
      Entry(const T& word, HashType hashValue) : word(word), hashValue(hashValue), next(nullptr) {}
   };

private:
   /// An instance of the hash funtion functor
   HashFunc hashFunc;
   /// The hashmap of pointers to actual entries
   std::vector<Entry*> entries;
   /// The pool allocator.
   Pool<Entry> entryAllocator;
   /// The exponent, see HashSet and summary for details
   uint64_t exp;
   /// The mask, see HashSet and summary for details
   uint64_t mask;
   /// The number of elements inside the hash,
   size_t size_;
   /// The size this hash map was allocated with
   uint64_t allocSize;

   public:
   /// Constructor
   MegaHashMap(size_t size) : entryAllocator() { hintSize(size); }
   /// Inserts an element into the hashtable. Returns the value for that key, regardless if an insertion happened or
   /// not. Hash value supplied externally
   Value* tryInsert(const T& word, HashType hashValue);
   /// Inserts an element into the hashtable. Returns the value for that key, regardless if an insertion happened or
   /// not.
   Value* tryInsert(const T& word) { return tryInsert(word, hashFunc(word)); }
   /// Finds an element inside the hashmap, returns the value if found or nullptr otherwise. Hash value supplied
   /// externally
   /// Inserts an element into the hashtable. Returns the value for that key, regardless if an insertion happened or
   /// not. Hash value supplied externally
   Value* tryInsertLocked(const T& word, HashType hashValue);
   Value* tryInsertLockfree(const T& word, HashType hashValue,Pool<Entry>& threadPrivatePool);

   Iterator find(const T& word, HashType hashValue) const;
   /// Finds an element inside the hashmap, returns the value if found or nullptr otherwise
   Iterator find(const T& word) const
   {
      if (size_ == 0) {
         return end();
      }
      else {
         return find(word, hashFunc(word));
      }
   }

   void erase(const T& word, HashType hashValue);
   void erase(const T& word)
   {
      if (size_ == 0) {
         return;
      }
      else {
         return erase(word, hashFunc(word));
      }
   }

   /// Alternative find: uses a type that can differ from T, e.g. T=string, T2=string_ref
   template <class T2> Value* altFind(const T2& word) { return altFind(word, hashFunc(word)); }
   /// Alternative find: uses a type that can differ from T, e.g. T=string, T2=string_ref
   template <class T2> Value* altFind(const T2& word, HashType hashValue);
   /// Pre allocates the hash to be able to contain size elements efficiently. Also, see summary and HashSet
   void hintSize(size_t size);
   /// Returns the current number of elements inside the hash
   size_t size() const { return size_; }
   /// Grows the hash to twice its capacity
   void grow() { grow(2 * capacity()); };
   /// Grows the hash to be able to efficiently contain 2* capacity elements
   void grow(size_t newSize);
   /// Clears all entries in the HashMap
   // void clear();
   /// Returns the current capacity of the hash
   size_t capacity() const { return entries.size(); };

   void prefetchBucketPtr(uint64_t hashValue) const { __builtin_prefetch(&entries[hashValue & mask], 0, 1); }
   void prefetchBucketEntry(uint64_t hashValue) const { __builtin_prefetch(entries[hashValue & mask], 0, 1); }


   // XXX: Custom code
   /// TODO: Disallow after migraiton
   MegaHashMap() : MegaHashMap(32){};
   HashType hash(const T& word) const { return hashFunc(word); }
   MegaHashMap& operator=(const MegaHashMap&) = delete;
   MegaHashMap(const MegaHashMap& other) = delete;

   MegaHashMap(MegaHashMap&& other)
      : hashFunc(std::move(other.hashFunc)), entries(std::move(other.entries)), entryAllocator(std::move(other.entryAllocator))
   {
      exp = other.exp;
      mask = other.mask;
      size_ = other.size_;
      allocSize = other.allocSize;
   }

   Iterator end() const {
       static const T end(nullptr,0);
       return Iterator(end, nullptr); }
   /// Returns 1 if the value is contained
   size_t count(const T& word) { return find(word) != nullptr ? 1 : 0; }
   /// Clears all entries in the map
   void clear();

   Value& operator[](const T& word)
   {
      auto match = find(word);
      if (match == end()) {
         auto val = tryInsert(word);
         new (val) Value();
         return *val;
      }
      else {
         return match->second;
      }
   }
};


/// Grows the hash to be able to efficiently contain 2* capacity elements
template <class T, class Value, class HashFunc> void MegaHashMap<T, Value, HashFunc>::grow(size_t newSize)
{
   allocSize = newSize;
   // Grow
   for (exp = 1; (size_t)(1ull << exp) < newSize; ++exp) {
   }
   // std::cerr << "Allocating " << allocSize << "::" << exp << "::" << (1ull << exp) << std::endl;
   std::vector<Entry*> newEntries(1ull << exp);
   mask = (1ull << exp) - 1ull;

   // Copy entries over
   for (Entry* entryChain : entries) {
      for (Entry *e = entryChain, *next; e != nullptr; e = next) {
         Entry*& entry = newEntries[e->hashValue & mask];
         if (entry == nullptr) { // immediate hit: use slot for e
            entry = e;
            next = e->next;
            e->next = nullptr;
         }
         else { // slot already in use: prepend e to chain
            next = e->next;
            e->next = entry;
            entry = e;
         }
      }
   }

   // Switch to new entry vector
   entries = std::move(newEntries);
}

template <typename Content> struct LockedPtr {
   std::atomic<Content*>& ref;
   Content* original;

   LockedPtr(Content*& contentPtrRef) : ref((std::atomic<Content*>&)contentPtrRef)
   {
      while (!acquireLock(ref)) {
      }
   }

   bool acquireLock(Content* ptr)
   {
      if (!isLocked(ptr) && ref.compare_exchange_weak(ptr, makeLocked(ptr))) {
         original = ptr;
         return true;
      }
      else {
         return false;
      }
   }

   void release() { ref = removeLock(ref); }

   static bool isLocked(Content* ptr) { return (((uintptr_t)ptr) & (lockMask)) != 0; }

   static Content* removeLock(Content* ptr) { return (Content*)(((uintptr_t)ptr) & (~lockMask)); }

   static Content* makeLocked(Content* ptr) { return (Content*)(((uintptr_t)ptr) | lockMask); }

   static constexpr uintptr_t lockMask = (1ull << ((sizeof(Content*) * 8) - 1));
};

template <class T, class Value, class HashFunc>
Value* MegaHashMap<T, Value, HashFunc>::tryInsertLockfree(const T& word, HashType hashValue,Pool<Entry>& threadPrivatePool)
{
   const auto asAtomic = [](Entry** entryPtrLoc) {
      return reinterpret_cast<std::atomic<Entry*>*>(entryPtrLoc);
   };
   const auto atomicInc = [](size_t& counter) {
      return reinterpret_cast<std::atomic<size_t>&>(counter).fetch_add(1);
   };

   std::atomic<Entry*>* entryPtrLoc = asAtomic(&entries[hashValue & mask]);
   Entry* entry = *entryPtrLoc;
   Entry* newEntry;
   // Unroll empty slot branch
   if (entry == nullptr) {
       newEntry = new (threadPrivatePool.allocate()) Entry(word, hashValue);
      if(!entryPtrLoc->compare_exchange_strong(entry, newEntry)){
          // another thread changed "entry", thus handle the chain
          goto chainCase;
      }
      atomicInc(size_);
      return &newEntry->value;
   }
   else {
      newEntry = new (threadPrivatePool.allocate()) Entry(word, hashValue);
   chainCase:
      do{
         // Unroll immediate hit branch
         if (entry->hashValue == hashValue && entry->word == word)
             return &entry->value;

         // Iterate to the end of the chain
         for (; entry->next != nullptr; entry = entry->next){
            Entry* eN = entry->next;
            if (eN->hashValue == hashValue && eN->word == word)
                return &eN->value;
         }

         // Word is not present, insert and return reference for backpatching
         entryPtrLoc = asAtomic(&entry->next);
         entry=nullptr;
      }while(!entryPtrLoc->compare_exchange_strong(entry, newEntry));

      atomicInc(size_);
      return &newEntry->value;
   }
}

/// Inserts an element into the hashtable. Returns the value for that key, regardless if an insertion happened or not.
template <class T, class Value, class HashFunc>
Value* MegaHashMap<T, Value, HashFunc>::tryInsertLocked(const T& word, HashType hashValue)
{
   static Value* allocatePtr = nullptr;

   if (capacity() <= (size_t)(size() * LoadFactor)) {
      grow();
   }

   LockedPtr<Entry> lockedPtr(entries[hashValue & mask]);

   Entry* entry = lockedPtr.original;
   // Unroll empty slot branch
   if (entry == nullptr) {
      ++size_;
      LockedPtr<Value> allocLock(allocatePtr);
      auto newEntry = entryAllocator.allocate();
      allocLock.release();
      new (newEntry) Entry(word, hashValue);
      auto value = &newEntry->value;
      entries[hashValue & mask] = newEntry;
      return value;
   }
   else {
      // Unroll immediate hit branch
      if (entry->hashValue == hashValue && entry->word == word) {
         auto value = &entry->value;
         lockedPtr.release();
         return value;
      }

      // Iterate to the end of the chain
      Entry* entryIter;
      for (entryIter = entry; entryIter->next != nullptr; entryIter = entryIter->next)
         if (entryIter->next->hashValue == hashValue && entryIter->next->word == word) {
            auto value = &entryIter->next->value;
            lockedPtr.release();
            return value;
         }

      // Word is not present, insert and return reference for backpatching
      ++size_;
      LockedPtr<Value> allocLock(allocatePtr);
      entryIter->next = entryAllocator.allocate();
      allocLock.release();
      new (entryIter->next) Entry(word, hashValue);
      auto value = &entryIter->next->value;
      lockedPtr.release();
      return value;
   }
}

/// Inserts an element into the hashtable. Returns the value for that key, regardless if an insertion happened or not.
template <class T, class Value, class HashFunc>
Value* MegaHashMap<T, Value, HashFunc>::tryInsert(const T& word, HashType hashValue)
{
   if (capacity() <= (size_t)(size() * LoadFactor)) {
      grow();
   }

   Entry*& entry = entries[hashValue & mask];
   // Unroll empty slot branch
   if (entry == nullptr) {
      ++size_;
      entry = entryAllocator.allocate();
      new (entry) Entry(word, hashValue);
      return &entry->value;
   }
   else {
      // Unroll immediate hit branch
      if (entry->hashValue == hashValue && entry->word == word)
         return &entry->value;

      // Iterate to the end of the chain
      Entry* entryIter;
      for (entryIter = entry; entryIter->next != nullptr; entryIter = entryIter->next)
         if (entryIter->next->hashValue == hashValue && entryIter->next->word == word)
            return &entryIter->next->value;

      // Word is not present, insert and return reference for backpatching
      ++size_;
      entryIter->next = entryAllocator.allocate();
      new (entryIter->next) Entry(word, hashValue);
      return &entryIter->next->value;
   }
}

/// Finds an element inside the MegaHashMap, returns the value if found or nullptr otherwise. Hash value supplied
/// externally
template <class T, class Value, class HashFunc>
typename MegaHashMap<T, Value, HashFunc>::Iterator MegaHashMap<T, Value, HashFunc>::find(const T& word, HashType hashValue) const
{
   if (size() == 0) {
      return Iterator(word, nullptr);
   }

   Entry* entry = entries[hashValue & mask];
   // Unroll empty slot branch
   if (entry == nullptr) {
      return Iterator(word, nullptr);
   }
   else {
      // Unroll immediate hit branch
      if (entry->hashValue == hashValue && entry->word == word)
         return Iterator(word, &entry->value);

      // Iterate to the end of the chain
      for (; entry->next != nullptr; entry = entry->next)
         if (entry->next->hashValue == hashValue && entry->next->word == word)
            return Iterator(word, &entry->next->value);

      return Iterator(word, nullptr);
   }
}

template <class T, class Value, class HashFunc>
void MegaHashMap<T, Value, HashFunc>::erase(const T& word, HashType hashValue)
{
   if (size() == 0) {
      return;
   }

   Entry* entry = entries[hashValue & mask];
   // Unroll empty slot branch
   if (entry == nullptr) {
      return;
   }
   else {
      // Unroll immediate hit branch
      if (entry->hashValue == hashValue && entry->word == word) {
         if (entry->next == nullptr) {
            entryAllocator.deallocate(entry);
            entries[hashValue & mask] = nullptr;
         }
         else {
            entry = entry->next;
         }
         return;
      }

      // Iterate to the end of the chain
      for (; entry->next != nullptr; entry = entry->next) {
         if (entry->next->hashValue == hashValue && entry->next->word == word) {
            entry->next = entry->next->next;
            return;
         }
      }
   }
}

/// Alternative find: uses a type that can differ from T, e.g. T=string, T2=string_ref
template <class T, class Value, class HashFunc>
template <class T2>
Value* MegaHashMap<T, Value, HashFunc>::altFind(const T2& word, HashType hashValue)
{
   Entry* entry = entries[hashValue & mask];
   // Unroll empty slot branch
   if (entry == nullptr) {
      return nullptr;
   }
   else {
      // Unroll immediate hit branch
      if (entry->hashValue == hashValue && entry->word == word)
         return &entry->value;

      // Iterate to the end of the chain
      for (; entry->next != nullptr; entry = entry->next)
         if (entry->next->hashValue == hashValue && entry->next->word == word)
            return &entry->next->value;

      return nullptr;
   }
}

/// Pre allocates the hash to be able to contain size elements efficiently. Also, see summary and HashSet
template <class T, class Value, class HashFunc> void MegaHashMap<T, Value, HashFunc>::hintSize(size_t size)
{
   this->allocSize = size;
   for (exp = 1; (size_t)(1ull << exp) < size; ++exp)
      ;
   entries.resize(1 << exp);
   mask = (1ull << exp) - 1ull;
   size_ = 0;
}

// XXX: Custom code

/// Clears all entries in the map
template <class T, class Value, class HashFunc> void MegaHashMap<T, Value, HashFunc>::clear()
{
   memset(entries.data(), 0, entries.size() * sizeof(Entry*));
   entryAllocator.reset();
   size_ = 0;
}
}
