#pragma once

#include "google/farmhash.hpp"
#include "immintrin.h"
#include "nmmintrin.h"
#include <algorithm>
#include <cassert>
#include <cstdint>
#include <cstring>
#include <string>

class StringPtrData {
   const char* ptr;
   size_t len;

   public:
   StringPtrData(const char* ptr, size_t len) : ptr(ptr), len(len) {}

   const char* data() const { return ptr; }

   size_t size() const { return len; }

   static void swap(StringPtrData& a, StringPtrData& b)
   {
      using std::swap; // bring in swap for built-in types

      swap(a.ptr, b.ptr);
      swap(a.len, b.len);
   }
};

template <typename Self> class BaseString {
   protected:
   StringPtrData strData;

   public:
   BaseString() : strData(nullptr, 0) {}
   BaseString(const char* ptr, size_t len) : strData(ptr, len) {}

   const char* data() const { return strData.data(); }

   size_t size() const { return strData.size(); }

   static void swap(Self& a, Self& b) { StringPtrData::swap(a.strData, b.strData); }

   const char* begin() const { return data(); }
   const char* end() const { return data() + size(); }

   std::string toStr() const { return std::string(data(), size()); }

   template <typename OtherSelf>
   inline int compare(size_t offset, size_t cmpLen, const BaseString<OtherSelf>& other) const
   {
      assert((offset + cmpLen) <= size());
      return cmp(data() + offset, cmpLen, other.data(), cmpLen);
   }

   Self& operator=(Self&& other)
   {
      strData = std::move(other.strData);
      return *reinterpret_cast<Self*>(this);
   }

   bool operator<(const Self& other) const { return cmp(data(), size(), other.data(), other.size()) < 0; }

   bool operator==(const Self& other) const { return matchSSE(data(), size(), other.data(), other.size()); }

   bool operator!=(const Self& other) const { return !matchSSE(data(), size(), other.data(), other.size()); }

   bool operator!=(const char other) const { return size() != 1 || *data() != other; }

   bool operator==(const char other) const { return size() == 1 && *data() == other; }

   uint8_t operator[](const size_t index) const
   {
      assert(index < size());
      return data()[index];
   }

   private:
   static int cmp(const char* a, size_t lenA, const char* b, size_t lenB)
   {
      if (lenA > lenB) {
         lenA = lenB;
      }

#pragma GCC ivdep
      for (size_t i = 0; i < lenA; i++) {
         if (a[i] != b[i]) {
            return b[i] - a[i];
         }
      }

      return (int32_t)lenA - (int32_t)lenB;
   }

   public:
   static bool matchSSE(const char* __restrict__ a, size_t lenA, const char* __restrict__ b, size_t lenB)
   {
      if (lenA != lenB) {
         return false;
      }

      auto length = lenA;
      const auto end = a + length;
      if (length < 16) {
         auto aReg = _mm_loadu_si128((const __m128i*)a);
         auto bReg = _mm_loadu_si128((const __m128i*)b);
         if (_mm_cmpestrc(aReg, length, bReg, length, _SIDD_UBYTE_OPS | _SIDD_CMP_EQUAL_EACH | _SIDD_NEGATIVE_POLARITY)) {
            return false;
         }
         else {
            return true;
         }
      }

      for (; a < end; a += 16, b += 16, length -= 16) {
         auto aReg = _mm_loadu_si128((const __m128i*)a);
         auto bReg = _mm_loadu_si128((const __m128i*)b);
         if (_mm_cmpestrc(aReg, length, bReg, length, _SIDD_UBYTE_OPS | _SIDD_CMP_EQUAL_EACH | _SIDD_NEGATIVE_POLARITY))
            return false;
      }
      return true;
   }

   private:
   static bool match(const char* __restrict__ a, size_t lenA, const char* __restrict__ b, size_t lenB)
   {
      if (lenA != lenB) {
         return false;
      }

#pragma GCC ivdep
      for (size_t i = 0; i < lenA; i++) {
         if (a[i] != b[i]) {
            return false;
         }
      }

      return true;
   }
};

class OwnedString : public BaseString<OwnedString> {
   public:
   OwnedString() : BaseString(nullptr, 0) {}

   OwnedString(const char* ptr, size_t len) : BaseString(ptr, len) {}

   OwnedString(OwnedString&& other) : BaseString(std::move(other)) {}

   OwnedString& operator=(OwnedString&& other)
   {
      strData = std::move(other.strData);
      return *this;
   }
   OwnedString& operator=(const OwnedString&) = delete;
   OwnedString(const OwnedString&) = delete;
};

class BorrowedString : public BaseString<BorrowedString> {
   public:
   BorrowedString() : BaseString(nullptr, 0) {}

   BorrowedString(const char* ptr, size_t len) : BaseString(ptr, len) {}

   inline BorrowedString substr(size_t pos) const
   {
      assert(pos <= size());
      return BorrowedString(data() + pos, size() - pos);
   }

   inline BorrowedString substr(size_t pos, size_t subLen) const
   {
      assert((pos + subLen) <= size());
      return BorrowedString(data() + pos, subLen);
   }

   inline std::pair<BorrowedString, BorrowedString> split(const char delim) const
   {
      if (size() == 0) {
         return std::make_pair(BorrowedString(), BorrowedString());
      }
      else {
         size_t pos = std::numeric_limits<size_t>::max();
         for (size_t i = 0; i < size(); i++) {
            if (data()[i] == delim) {
               pos = i;
               break;
            }
         }

         if (pos == std::numeric_limits<size_t>::max()) {
            pos = size();
         }

         auto firstPart = substr(0, pos);
         auto secondPart = [this, pos]() {
            if (pos == size()) {
               return BorrowedString();
            }
            else {
               return substr(pos);
            }
         }();

         return std::make_pair(firstPart, secondPart);
      }
   }

   inline OwnedString toOwned() const
   {
      auto ownedPtr = new char[size()];
      memcpy(ownedPtr, data(), size());
      return OwnedString(ownedPtr, size());
   }

   static inline BorrowedString fromOwned(const OwnedString& other)
   {
      return BorrowedString(other.data(), other.size());
   }

   inline const OwnedString& toOwnedRef() const { return (const OwnedString&)(*this); }

   static inline const BorrowedString& refFromOwned(const OwnedString& other) { return (const BorrowedString&)(other); }
};

template <typename Func> void processStringSplits(const BorrowedString& input, const char delim, Func func)
{
   auto complete = input;
   while (true) {
      auto tokens = complete.split(delim);
      if (tokens.first.size() > 0) {
         func(complete, tokens.first, tokens.second);
      }
      if (tokens.second.size() > 0) {
         complete = tokens.second.substr(1);
      }
      else {
         break;
      }
   }
}

namespace tokenize {
template <typename Func> void wordSplits(const BorrowedString& input, const char delim, Func func)
{
   auto start = input.data();
   const auto end = input.data() + input.size();
   auto current = start;
   for (; current < end; ++current) {
      if (*current == delim) {
         BorrowedString word(start, current - start);
         func(word);
         start = current + 1;
      }
   }

   if (start < end) {
      BorrowedString word(start, end - start);
      func(word);
   }
}

template <typename Func> void wordSplitOffsets(const BorrowedString& input, const char delim, Func func)
{
   auto start = input.data();
   const auto end = input.data() + input.size();
   auto current = start;
   for (; current < end; ++current) {
      if (*current == delim) {
         func(current);
         start = current + 1;
      }
   }

   if (start < end) {
      func(end);
   }
}

const char was[16] = {' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' '};
const __m128i whitespaces = _mm_loadu_si128((const __m128i*)was);

template <typename Func> void wordSplitOffsetsSSE(const BorrowedString& input, const char, Func func)
{
   auto start = input.data();
   const auto end = input.data() + input.size();
   auto found = start;

   for (; start < end;) {
      auto chars = _mm_loadu_si128((const __m128i*)start);
      uint32_t hit = _mm_cmpistri(chars, whitespaces, _SIDD_CMP_EQUAL_EACH | _SIDD_UBYTE_OPS | _SIDD_LEAST_SIGNIFICANT);
      start += hit;
      if (hit < 16 && start < end) {
         found = start;
         start++;
         func(found);
      }
   }

   if (found < end) {
      func(end);
   }
}

const char nls[16] = {'\n', '\n', '\n', '\n', '\n', '\n', '\n', '\n', '\n', '\n', '\n', '\n', '\n', '\n', '\n', '\n'};
const __m128i newlines = _mm_loadu_si128((const __m128i*)nls);

inline const char* nextLinebreak(const char* __restrict__ start, const char* __restrict__ end)
{
   for (; start < end; start += 16) {
      auto chars = _mm_loadu_si128((const __m128i*)start);
      uint32_t hit = _mm_cmpistri(chars, newlines, _SIDD_CMP_EQUAL_EACH | _SIDD_UBYTE_OPS | _SIDD_LEAST_SIGNIFICANT);
      if (hit < 16) {
         if ((start + hit) < end) {
            return start + hit;
         }
         else {
            return end;
         }
      }
   }
   return end;
}
}

// Tested also https://github.com/rurban/smhasher/blob/master/crc32_hw.c but was slower
// Some primes between 2^63 and 2^64 for various uses.
// from city hash
static const uint64_t k0 = 0xc3a5c85c97cb3127ULL;
static const uint64_t k1 = 0xb492b66fbe98f273ULL;
static inline uint64_t crc32_hash_rowdy(const char* start, uint64_t len)
{
   uint64_t a = k0 & 0xFFFFFFFF;
   const uint64_t* bulk = reinterpret_cast<const uint64_t*>(start);
   for (auto end = bulk + (len / 8); bulk < end; ++bulk) {
      a = _mm_crc32_u64(a, *bulk);
   }
   auto rest = len & 7;
   if (rest) {
      auto mask = (1ull << (rest * 8)) - 1;
      auto inp = *bulk & mask;
      a = _mm_crc32_u64(a, inp);
   }
   return a * k1;
}

namespace std {
template <> struct hash<BorrowedString> {
   size_t operator()(const BorrowedString& x) const noexcept { return crc32_hash_rowdy(x.data(), x.size()); }
};
}

namespace std {
template <> struct hash<OwnedString> {
   size_t operator()(const OwnedString& x) const noexcept { return crc32_hash_rowdy(x.data(), x.size()); }
};
}

// DANGEROUS!
inline const BorrowedString& strToBorrowed(const std::string& str)
{
   auto bstr = new BorrowedString(str.data(), str.size());
   return *bstr;
}
