#pragma once

#include "util/string.hpp"
#include <string>
#include <mutex>
#include <vector>

class BufferedInput {
   static const size_t chunkSize=1024*16;
   std::vector<char> buffer;
   std::mutex getline_mtx;
   std::mutex fill_mtx;
   size_t available;
   size_t offset;
   bool eof;

   public:
   BufferedInput(size_t size) : buffer(size), available(0), offset(0), eof(false) {}


inline bool getline(BorrowedString& nextLine)
{
   auto start = offset;
   auto current = start;
   const char* ptr = buffer.data();

   while (true) {
      if (current == available) {
         fill();
         ptr = buffer.data();
      }
      current = tokenize::nextLinebreak(ptr + current, ptr + available) - ptr;
      if (current < available || eof) {
         BorrowedString line(ptr + start, current - start);
         offset = current + 1;
         BorrowedString::swap(nextLine, line);
         return !eof;
      }
   }
}
   template<typename L>
   void eachLine(L l);
   void reset();

   private:
   void fill();
   size_t fillChunk(uint64_t size);
};

template<typename L>
void BufferedInput::eachLine(L l){
    bool flushing = false;
    while(true){
        size_t retrieved;
        {
            if(!flushing)fill_mtx.lock();
            if(eof) return;
            retrieved = fillChunk(chunkSize);
            if(!flushing)getline_mtx.lock();
            //exclusive access to BufferedInput
            available+=retrieved;
            if(retrieved ==0){
                eof=true;
                fill_mtx.unlock();
                getline_mtx.unlock();
                return;
            }
            auto end = buffer.data()+available;
            if(end[-3]=='\n'&&end[-2]=='F'&&end[-1]=='\n')
                flushing=true; //keep fill_mtx in case we are flushing
            else {
                flushing = false;
                fill_mtx.unlock();
            }
        }

        auto start = offset;
        auto current = start;
        const char* ptr = buffer.data();

        current = tokenize::nextLinebreak(ptr + current, ptr + available) - ptr;
        for (;current != available; current = tokenize::nextLinebreak(ptr + start, ptr + available) - ptr) {
            BorrowedString line(ptr + start, current - start);
            l(line);
            offset = current + 1;
            start=current+1;
        }
        if(!flushing) getline_mtx.unlock();
        else{
            reset();
        }
    }
}
