#pragma once

#include <algorithm>
#include <atomic>
#include <condition_variable>
#include <iostream>
#include <mutex>
#include <thread>
#include <vector>
#include "util/log.hpp"
#include <cassert>
#include <pthread.h>


namespace workerPool {
void pinThread(size_t threadId);
void pin(size_t hwThreadId, size_t threadId);
uint32_t nextHyperThreadOf(uint32_t threadId);
size_t getWorkerThreadId();
void yield(unsigned k);
}

class WorkerPool {
   std::vector<std::thread> threads;

   std::atomic<bool> running;
   std::atomic<size_t> activeWorkers;
   std::atomic<size_t> currentWorkVersion;
   std::function<void(size_t)> currentWork;

   std::atomic<bool> blocked;
   std::condition_variable blockedCond;
   std::mutex blockedMutex;

   public:
   WorkerPool(size_t numWorkers);
   ~WorkerPool();

   void submit(std::function<void(size_t)> function);
   void waitForCompletition();
   void block();
   void unblock();
   size_t size() { return threads.size(); }
};

size_t getWorkerThreadId();

template <typename Function> void parallel_for(WorkerPool& pool, size_t start, size_t end, Function work,const size_t minChunkSize=1)
{
   if(end-start == 1) {
      for (size_t i = start; i < end; i++) {
         work(i);
      }
      return;
   }
   std::atomic<size_t> counter(start);
   auto totalWorkers = pool.size() + 1;
   auto threadWorkFn = [&pool, &counter, totalWorkers, start, end, work,minChunkSize](size_t /*threadId*/) {
      auto total = end - start;
      auto chunkSize = std::max(minChunkSize, total / (totalWorkers * 8));
      size_t currentOffset;
      while ((currentOffset = counter.fetch_add(chunkSize)) < end) {
         auto currentLimit = std::min(end, currentOffset + chunkSize);
         for (size_t i = currentOffset; i < currentLimit; i++) {
            work(i);
         }
      }
   };

   pool.submit(threadWorkFn);
   threadWorkFn(pool.size());
   pool.waitForCompletition();
}

template <typename Function> void parallel_each_thread(WorkerPool& pool, Function work)
{
   auto threadWorkFn = [&pool, work](size_t threadId) {
      work(threadId);
   };

   pool.submit(threadWorkFn);
   threadWorkFn(pool.size());
   pool.waitForCompletition();
}
