#pragma once

#include "util/basengramer.hpp"
#include "util/output.hpp"
#include "util/sectiontimer.hpp"
#include "util/string.hpp"
#include "util/workloadstats.hpp"

template <bool detailedStats, bool timingStats, typename NgramProcessor> class StatNgramer {
   NgramProcessor& processor;
   SectionTimer& timer;
   WorkloadStats& workloadStats;

   public:
   StatNgramer(NgramProcessor& processor, SectionTimer& timer, WorkloadStats& workloadStats)
      : processor(processor), timer(timer), workloadStats(workloadStats)
   {
   }

   void query(const BorrowedString& document, QueryOut& out)
   {
      processor.query(document, out);
      if (detailedStats) {
         workloadStats.analyzeQuery(document.toStr(), out.toString());
      }
      if (timingStats) {
         timer.traceQuery();
      }
   }

   void add(const BorrowedString& ngram)
   {
      processor.add(ngram);
      if (detailedStats) {
         workloadStats.analyzeAddition(ngram.toStr());
      }
      if (timingStats) {
         timer.traceAddition();
      }
   }

   void remove(const BorrowedString& ngram)
   {
      processor.remove(ngram);
      if (detailedStats) {
         workloadStats.analyzeRemove(ngram.toStr());
      }
      if (timingStats) {
         timer.traceRemove();
      }
   }
};

template <bool detailedStats, bool timingStats, typename NgramProcessor> class MTStatNgramer {
   public:
   NgramProcessor& processor;
   SectionTimer& timer;
   private:
   WorkloadStats& workloadStats;

   public:
   MTStatNgramer(NgramProcessor& processor, SectionTimer& timer, WorkloadStats& workloadStats)
      : processor(processor), timer(timer), workloadStats(workloadStats)
   {
   }

   void query(const BaseNgramer::SequenceId id, const BorrowedString& document, QueryOut& out)
   {
      processor.query(id, document, out);
      if (detailedStats) {
         workloadStats.analyzeQuery(document.toStr(), out.toString());
      }
      if (timingStats) {
         timer.traceQuery();
      }
   }

   void add(const BaseNgramer::SequenceId id, const BorrowedString& ngram, const uint32_t patternId)
   {
      processor.add(id, ngram, patternId);
      if (detailedStats) {
         workloadStats.analyzeAddition(ngram.toStr());
      }
      if (timingStats) {
         timer.traceAddition();
      }
   }

   void remove(const BaseNgramer::SequenceId id, const BorrowedString& ngram)
   {
      processor.remove(id, ngram);
      if (detailedStats) {
         workloadStats.analyzeRemove(ngram.toStr());
      }
      if (timingStats) {
         timer.traceRemove();
      }
   }
};
