#pragma once

#include <algorithm>
#include <atomic>
#include <condition_variable>
#include <iostream>
#include <mutex>
#include <thread>
#include <vector>
#include "nmmintrin.h"
#include "immintrin.h"
#include <tuple>
#include "util/basengramer.hpp"
#include "util/workerpool.hpp"
#include "util/log.hpp"
#include "shared.hpp"


struct PoolOperation {
    BorrowedString string; // char*, size_t
    BaseNgramer::SequenceId id; // uint32_t
    uint32_t opCounter: 24, partitionId: 8;
    uint32_t patternOrQueryId; // or REMOVE_MARKER

    static const uint8_t SINGLETHREADED_MARKER=~0;

    PoolOperation() {} 
    PoolOperation(BaseNgramer::SequenceId id, BorrowedString string, unsigned opCounter, uint32_t patternOrQueryId) : string(std::move(string)), id(id), opCounter(opCounter), patternOrQueryId(patternOrQueryId) {}
    void markAsSingleThreaded() {
       partitionId=SINGLETHREADED_MARKER;
    }
    bool isSingleThreaded() { return partitionId==SINGLETHREADED_MARKER;}
};

static const unsigned OPERATION_BUFFER_SIZE=20*1024;
static const unsigned CHUNK_SIZE=8;
static const unsigned INTRA_QUERY_THRESHOLD=16;
static const unsigned MIN_SIZE_INTRA_QUERY=20000;
template<typename T>
struct WorkQueue {
   T elements[OPERATION_BUFFER_SIZE];
   alignas(64) std::atomic<unsigned> scheduled{0};
   alignas(64) std::atomic<unsigned> added{0};
   alignas(64) std::atomic<unsigned> completed{0};

   bool pop(T*& elem) {
      auto index=scheduled.load();
      do {
         if(index >= added) {
            return false;
         }
      } while(!scheduled.compare_exchange_strong(index, index+1));
      elem = &elements[index];
      return true;
   }


   unsigned multiPop(T*& elem) {
      unsigned index=scheduled.load();
      unsigned newScheduled;
      do {
         if(index>=added) return 0;
         newScheduled = std::min(added.load(),index+CHUNK_SIZE);
         assert(index <= added);
      } while(!scheduled.compare_exchange_weak(index, newScheduled));
      elem = &elements[index];
      assert(newScheduled-index <= CHUNK_SIZE);
      return newScheduled-index;
   }

   void reset() {
      added=0;
      scheduled=0;
      completed=0;
   }

   unsigned size() {
      return added;
   }


   bool isCompleted() {
      return completed==added;
   }

   void wait() {
      while(!isCompleted()) {
         sched_yield();
      }
   }

   bool hasCapacity(size_t additional) {
      return (added+additional) < OPERATION_BUFFER_SIZE;
   }

   void push(T& elem) {
       elements[added]=std::move(elem);
       added++;
   }

    void multiPush(T* batch, size_t size) {
        for (size_t i = 0; i < size; i++) {
            T& elem = batch[i];
            elements[added + i]=std::move(elem);
        }
        added += size;
    }
};

const char zeros[16] = {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};
const __m128i zerosRegister = _mm_loadu_si128((const __m128i*)zeros);


template <typename NgramProcessor>
class PriorityWorkerPool {

   enum Operation {add, query, remove};


   std::atomic<bool> running;
   std::vector<std::thread> threads;



   NgramProcessor& ngramer;

   QueryOut queryOuts[OPERATION_BUFFER_SIZE+1];
   typedef uint8_t progress;
   progress progressVector[OPERATION_BUFFER_SIZE];
   struct ProgressCounter {
      alignas(64) uint64_t v{0};
   };

   std::vector<ProgressCounter> progressCounters;
   alignas(64) std::atomic<unsigned> queriesPrinted{0};
   alignas(64)std::atomic<unsigned> queriesAdded{0};

   typedef PoolOperation operationInfo;
   typedef WorkQueue<operationInfo> operationList;

public:
   operationList queryOperations;
   operationList addsAndRemoves;

private:
   bool waitForProgress(uint32_t requiredProgress, uint64_t& localProgressCounter){
      unsigned spins=0;
      while(requiredProgress > localProgressCounter){
         //localProgressCounter+=progressVector[localProgressCounter];
         auto bytes = _mm_loadu_si128((const __m128i*)&progressVector[localProgressCounter]);
         auto hits= _mm_cmpistri(bytes,zerosRegister, _SIDD_CMP_EQUAL_EACH|_SIDD_UBYTE_OPS|_SIDD_LEAST_SIGNIFICANT);
         localProgressCounter +=hits;
         if(spins++ > 10) return false;
      }
      return true;
   }

   bool executeAddOrRemove();

    bool executeQuery(uint64_t& localProgressCounter);

   public:
    const uint32_t REMOVE_MARKER=~0;
    unsigned opCounter=0;

    PriorityWorkerPool(size_t numWorkers, NgramProcessor& ngramer);

   ~PriorityWorkerPool();

   void writeResults();
    size_t size() { return threads.size(); }

    void scheduleAdd(PoolOperation poolOperation) {
        addsAndRemoves.push(poolOperation);
    }
    void scheduleRemove(PoolOperation poolOperation) {
        addsAndRemoves.push(poolOperation);
    }
    bool reachedInterQueryThreshold=false;
    void scheduleQuery(PoolOperation poolOperation) {
       if(reachedInterQueryThreshold || queryOperations.added > INTRA_QUERY_THRESHOLD){
          reachedInterQueryThreshold=true;
          PoolOperation pCopy = poolOperation;
          pCopy.markAsSingleThreaded();
          queryOperations.push(pCopy);
       } else {
          queryOperations.push(poolOperation);
          assert(poolOperation.partitionId ==0);
          for(unsigned pId=1; pId<QUERY_PARTITIONS; ++pId){
             PoolOperation pCopy = poolOperation;
             pCopy.partitionId=pId;
             queryOperations.push(pCopy);
          }
       }
       queriesAdded++;
    }
    void scheduleAddOrRemoveBatched(PoolOperation* batch, size_t size) {
        addsAndRemoves.multiPush(batch, size);
    }
};

template <typename NgramProcessor>
bool PriorityWorkerPool<NgramProcessor>::executeQuery(uint64_t& localProgressCounter) {
   operationInfo* opInfo;
   if(queryOperations.pop(opInfo)){
      uint32_t requiredProgress=opInfo->opCounter;
      while(!waitForProgress(requiredProgress,localProgressCounter)){
         //if no progress is coming: make some
         executeAddOrRemove();
      }
      auto& queryOut = queryOuts[opInfo->patternOrQueryId];
       this->ngramer.intraQueryRange(opInfo->id, opInfo->string, queryOut, opInfo->partitionId);
       if(!opInfo->isSingleThreaded()) {
         queryOut.mergePartitions();
      }

      return true;
   }
   return false;
}

template <typename NgramProcessor>
bool PriorityWorkerPool<NgramProcessor>::executeAddOrRemove() {
   operationInfo* opInfo;
   if(auto count=addsAndRemoves.multiPop(opInfo)) {
      size_t adds=0;
      for(auto iter=opInfo,limit=iter+count;iter<limit;++iter){
          uint32_t patternId = iter->patternOrQueryId;
         if(patternId!=REMOVE_MARKER)
              this->ngramer.add(iter->id, iter->string, adds, patternId);
         else
              this->ngramer.remove(iter->id, iter->string, adds);
      }
      ngramer.accountAdds(adds);
      if(count==CHUNK_SIZE){
          memset(&progressVector[opInfo->opCounter], 1, sizeof(progress)*CHUNK_SIZE);
      } else {
         memset(&progressVector[opInfo->opCounter], 1, sizeof(progress)*count);
      }
      addsAndRemoves.completed+=count;
      return true;
   }
   return false;
}


template <typename NgramProcessor>
PriorityWorkerPool<NgramProcessor>::PriorityWorkerPool(size_t numWorkers, NgramProcessor& ngramer)
   : running(true), ngramer(ngramer)
{
   auto earlyOutput = [this]() {
      auto numQueries = queriesAdded.load();//queryOperations.size();
      unsigned queriesPrintedLocal = 0;
      //                                       keep this comparison < (not !=)         vvvv
      // otherwise, bad things will happen (to you)
      // Bad things: queriesPrinted and numQueries are both written to by the thread that
      // calls write Results. One possible interleaving: numQueries = 0, this threads reads new
      // value of numQueries, but not new value of queriesPrinted. Thus iter > limit which
      // means iter != limit --> Boom
      for (auto iter = queryOuts + queriesPrinted, limit = queryOuts + numQueries; iter < limit;) {
         assert(limit > iter);
         if (iter->isDone()) {
            iter->write();
            iter->reset();
            ++iter;
            queriesPrintedLocal++;
         }
         else {
            break;
         }
      }
      if (queriesPrintedLocal) {
         queriesPrinted += queriesPrintedLocal;
      }
   };

   memset(progressVector, 0, sizeof(progressVector));
   progressCounters.resize(numWorkers);
   for (size_t i = 0; i < numWorkers; i++) {
      auto id = i;
      threads.emplace_back([this, id, earlyOutput, numWorkers]() {
         workerPool::pinThread(id);
         uint64_t& localProgressCounter = progressCounters[id].v;
         while (running) {
            if(id==0) {
               earlyOutput();
            }

            if (executeAddOrRemove()) {
               continue;
            }

            if (executeQuery(localProgressCounter)) {
               continue;
            }

            // workerPool::yield(k++);
            __asm__("pause;");
            __asm__("pause;");
            __asm__("pause;");
            __asm__("pause;");
         }
      });
   }

   // Pin main thread also
   workerPool::pinThread(numWorkers);
}

template <typename NgramProcessor>
PriorityWorkerPool<NgramProcessor>::~PriorityWorkerPool()
{
   running = false;
   for (auto& thread : threads) {
      thread.join();
   }
}

template <typename NgramProcessor>
void PriorityWorkerPool<NgramProcessor>::writeResults()
{
   uint64_t localProgressCounter = 0;
   auto numQueries=queriesAdded.load();
   while(queriesPrinted < numQueries) {
      if (executeAddOrRemove()) {
         continue;
      }

      if (executeQuery(localProgressCounter)) {
         continue;
      }

      __asm__ ( "pause;" );
      __asm__ ( "pause;" );
      __asm__ ( "pause;" );
      __asm__ ( "pause;" );
   }

   fflush(stdout);

   queryOperations.reset();
   addsAndRemoves.wait();
   queriesPrinted=0;
   queriesAdded=0;

   addsAndRemoves.reset();
   memset(&progressVector,0,opCounter*sizeof(progress));
   for(unsigned i=0; i<progressCounters.size();++i){
      progressCounters[i].v=0;
   }
   opCounter=0;
   reachedInterQueryThreshold=false;
}
