#include "singlethreaded/awesomengramer.hpp"
#include "util/shared.hpp"
#include <algorithm>
#include <iostream>
#include <iterator>
#include <sstream>
#include <unordered_set>

template <typename HashType> void combineHash(HashType& totalHash, HashType wordHash)
{
   totalHash = (totalHash << 2) + wordHash;
}

void STAwesomeNgramer::query(const BorrowedString& document, QueryOut& out)
{
   words.reserve(document.size());
   wordHashes.reserve(words.size());

   wordHashes.emplace_back(0);
   words.emplace_back(document.data() - 1);
   tokenize::wordSplitOffsetsSSE(document, ' ', [this](const char* wordEnd) {
      words.emplace_back(wordEnd);
      const auto wordStart = words[words.size() - 2] + 1;
      auto word = BorrowedString(wordStart, wordEnd - wordStart);
      wordHashes.emplace_back(std::hash<BorrowedString>()(word));
   });

   const auto localVersion = ++version;
   for (size_t i = 1; i < words.size(); i++) {
      patterns.prefetchBucketPtr(wordHashes[i + 12]);
      patterns.prefetchBucketEntry(wordHashes[i + 6]);
      const auto wordStart = words[i - 1] + 1;
      auto offset = i;
      size_t suffixLength = 0;
      size_t minSuffixLength = 1;
      size_t numWords = 0;
      size_t skipLength = 1;
      HashType runningHash = 0;
      do {
         numWords++;
         const auto wordEnd = words[offset];
         const auto word = BorrowedString(wordStart, wordEnd - wordStart);
         combineHash<HashType>(runningHash, wordHashes[offset]);
         // std::cerr<<minSuffixLength<<"::"<<word.toStr()<<"::"<<numWords<<std::endl;
         if (numWords == skipLength || numWords == minSuffixLength) {
            auto match = patterns.altFind(word, runningHash);
            if (match != nullptr) {
               auto& entry = *match;
               suffixLength = entry.longestSuffix;
               minSuffixLength = entry.shortestSuffix;
               skipLength <<= 1;
               if (entry.version < localVersion) {
                  entry.version = localVersion;
                  out.addNgram(word);
               }
            }
            else {
               break;
            }
         }
         offset++;
      } while (numWords < suffixLength && offset < words.size());
   }

   words.clear();
   wordHashes.clear();
   out.finishResult();
}

void STAwesomeNgramer::add(const BorrowedString& ngram)
{
   auto borrowedStoredNgram = arena.intern(ngram);

   words.emplace_back(borrowedStoredNgram.data() - 1);
   wordHashes.emplace_back(0);
   {
      HashType runningHash = 0;
      tokenize::wordSplitOffsetsSSE(borrowedStoredNgram, ' ', [this, &runningHash](const char* prefix) {
         words.emplace_back(prefix);
         auto i = words.size() - 1;
         auto word = BorrowedString(words[i - 1] + 1, words[i] - words[i - 1] - 1);
         combineHash<HashType>(runningHash, std::hash<BorrowedString>()(word));
         wordHashes.emplace_back(runningHash);
         patterns.prefetchBucketPtr(runningHash);
      });
   }

   const auto length = words.size() - 1;

   for (size_t i = 1; i < words.size(); i++) {
      auto prefix = borrowedStoredNgram.substr(0, words[i] - borrowedStoredNgram.data());
      auto& info = *patterns.tryInsert(SmallString<>(prefix), wordHashes[i]);
      if (info.longestSuffix < length) {
         info.longestSuffix = length;
      }
      if (info.shortestSuffix > length && i != length) {
         info.shortestSuffix = length;
      }
      if (i == length) {
         info.version = version;
      }
   }

   words.clear();
   wordHashes.clear();
}

void STAwesomeNgramer::remove(const BorrowedString& ngram)
{
   // Compute hash of word
   const char* start = ngram.data() - 1;
   HashType runningHash = 0;
   tokenize::wordSplitOffsets(ngram, ' ', [this, &start, &runningHash](const char* end) {
      auto word = BorrowedString(start + 1, end - start - 1);
      combineHash<HashType>(runningHash, std::hash<BorrowedString>()(word));
      start = end;
   });

   patterns.tryInsert(SmallString<>(ngram), runningHash)->invalidate();
}

void STAwesomeNgramer::finalizeLoading()
{
   patterns.grow();
   patterns.grow();
   patterns.grow();
   patterns.grow();
}
